# main.py (remplacer intégralement)
import sys
import os
import traceback
from PyQt6.QtWidgets import (
    QApplication, QWidget, QLabel, QVBoxLayout, QProgressBar, QGraphicsOpacityEffect, QMessageBox
)
from PyQt6.QtGui import QPixmap, QFont, QColor, QPalette
from PyQt6.QtCore import Qt, QTimer, QPropertyAnimation

# Import de la page de login (doit exister)
try:
    from gui.login_page import LoginPage
except Exception as e:
    # Si l'import échoue, on lève une erreur explicite et on loggue
    print("Erreur import gui.login_page:", e)
    traceback.print_exc()
    # On re-raise pour que le handler global capture ça aussi
    raise

BASE_DIR = os.getcwd()
LOGO2_PATH = os.path.join(BASE_DIR, "logo2.png")  # chemin logo

# --- handler global exceptions (évite fermeture silencieuse)
def global_except_hook(exctype, value, tb):
    err = "".join(traceback.format_exception(exctype, value, tb))
    # écrire un fichier log simple
    try:
        log_path = os.path.join(BASE_DIR, "app_error.log")
        with open(log_path, "a", encoding="utf-8") as f:
            f.write("\n\n" + ("="*60) + "\n")
            f.write(err)
    except Exception:
        pass
    # afficher à l'utilisateur (ne tue pas automatiquement le processus ici)
    try:
        QMessageBox.critical(None, "Erreur fatale", f"Une erreur inattendue est survenue:\n{value}\n\nVoir app_error.log")
    except Exception:
        # si QMessageBox indisponible (thread non-GUI), on imprime juste
        print("Erreur fatale (QMessageBox indisponible):", value)
    # appeler le hook par défaut (termine l'application)
    sys.__excepthook__(exctype, value, tb)

sys.excepthook = global_except_hook

class SplashScreen(QWidget):
    def __init__(self, duration=4000):
        super().__init__()
        self.duration = duration
        self.setWindowFlags(Qt.WindowType.FramelessWindowHint)
        self.setFixedSize(450, 350)

        # Fond blanc
        palette = self.palette()
        palette.setColor(QPalette.ColorRole.Window, QColor("#FFFFFF"))
        self.setPalette(palette)
        self.setAutoFillBackground(True)

        main_layout = QVBoxLayout()
        main_layout.setContentsMargins(0,0,0,0)
        main_layout.setSpacing(0)
        self.setLayout(main_layout)

        # Bande bleu en haut
        top_bar = QWidget()
        top_bar.setFixedHeight(50)
        top_bar.setStyleSheet("background-color: #1A2A4D;")
        main_layout.addWidget(top_bar)

        # Logo centré
        self.logo_label = QLabel()
        if os.path.exists(LOGO2_PATH):
            pixmap = QPixmap(LOGO2_PATH).scaled(200, 200, Qt.AspectRatioMode.KeepAspectRatio,
                                                Qt.TransformationMode.SmoothTransformation)
            self.logo_label.setPixmap(pixmap)
        self.logo_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        main_layout.addWidget(self.logo_label, stretch=1)

        # Bande bleu en bas
        bottom_bar = QWidget()
        bottom_bar.setFixedHeight(60)
        bottom_bar.setStyleSheet("background-color: #1A2A4D;")
        bottom_layout = QVBoxLayout()
        bottom_layout.setContentsMargins(10,5,10,5)
        bottom_layout.setSpacing(5)
        bottom_bar.setLayout(bottom_layout)

        footer_label = QLabel("D-Square Technologies | Application by Musagara Daniel")
        footer_label.setFont(QFont("Segoe UI", 10, QFont.Weight.Bold))
        footer_label.setStyleSheet("color: #FFFFFF;")
        footer_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        bottom_layout.addWidget(footer_label)

        # Barre de progression
        self.progress = QProgressBar()
        self.progress.setRange(0, 100)
        self.progress.setValue(0)
        self.progress.setTextVisible(False)
        self.progress.setStyleSheet("""
            QProgressBar {
                border: 1px solid #FFFFFF;
                border-radius: 5px;
                background-color: #1a1a1a;
            }
            QProgressBar::chunk {
                background-color: #00BFFF;
            }
        """)
        bottom_layout.addWidget(self.progress)
        main_layout.addWidget(bottom_bar)

        # Effets de fondu
        self.logo_opacity = QGraphicsOpacityEffect()
        self.logo_label.setGraphicsEffect(self.logo_opacity)
        self.logo_anim = QPropertyAnimation(self.logo_opacity, b"opacity")
        self.logo_anim.setStartValue(0)
        self.logo_anim.setEndValue(1)
        self.logo_anim.setDuration(duration)
        self.logo_anim.start()

        self.progress_opacity = QGraphicsOpacityEffect()
        self.progress.setGraphicsEffect(self.progress_opacity)
        self.progress_anim = QPropertyAnimation(self.progress_opacity, b"opacity")
        self.progress_anim.setStartValue(0)
        self.progress_anim.setEndValue(1)
        self.progress_anim.setDuration(duration)
        self.progress_anim.start()

        # Timer pour la progression
        # On utilise interval = max(1, duration//100) pour éviter interval 0
        interval = max(1, duration // 100)
        self.timer = QTimer(self)
        self.timer.timeout.connect(self.handle_timer)
        self.timer.start(interval)
        self.step = 0

    def handle_timer(self):
        self.step += 1
        # clamp step
        if self.step > 100:
            self.step = 100
        self.progress.setValue(self.step)
        if self.step >= 100:
            # On arrête le timer et on cache (hide) le splash.
            self.timer.stop()
            # hide au lieu de close pour éviter de "fermer" la fenêtre
            self.hide()

# ----------------- Lancement de l'app -----------------
if __name__ == "__main__":
    # Création de l'application
    app = QApplication(sys.argv)

    # IMPORTANT: Empêcher la fermeture automatique si toutes fenêtres fermées
    # (utile pendant debug / si une fenêtre est fermée involontairement).
    # Si tu préfères comportement normal (process termine à la fermeture), mets à True.
    app.setQuitOnLastWindowClosed(False)

    # Crée et montre le splash
    splash = SplashScreen(duration=4000)
    splash.show()

    # Pré-crée le login globalement pour qu'il reste en mémoire
    # (évite que l'objet soit GC et que la fenêtre disparaisse).
    try:
        login_window = LoginPage()
    except Exception as e:
        # problème d'import ou d'initialisation: log et message
        print("Erreur initialisation LoginPage:", e)
        traceback.print_exc()
        QMessageBox.critical(None, "Erreur", f"Impossible d'initialiser l'interface de connexion:\n{e}")
        # on continue pour permettre au hook global de gérer si besoin
        login_window = None

    # Fonction sûre pour afficher le login après le splash
    def show_login():
        try:
            # s'assurer que splash est caché (déjà caché par handle_timer mais on double-check)
            try:
                splash.hide()
            except Exception:
                pass
            if login_window:
                login_window.show()
            else:
                QMessageBox.critical(None, "Erreur", "Login non disponible. Voir log.")
        except Exception as e:
            print("Erreur show_login:", e)
            traceback.print_exc()

    # On déclenche l'affichage du login après la durée voulue.
    # On aligne le singleShot sur la durée du splash
    QTimer.singleShot(4000, show_login)

    # Lance la boucle principale (protégée)
    try:
        exit_code = app.exec()
    except Exception as e:
        # catch improbable: log
        print("Erreur lors de exec():", e)
        traceback.print_exc()
        exit_code = 1
    # assure une sortie propre
    sys.exit(exit_code)
