import os
import qrcode
from PIL import Image, ImageDraw, ImageFont
from datetime import datetime
from reportlab.lib.pagesizes import A4
from reportlab.lib.units import mm
from reportlab.pdfgen import canvas

# -----------------------------
# --- CONFIGURATION Dossiers ---
# -----------------------------
BASE_DIR = os.path.dirname(os.path.abspath(__file__))  # dossier du script
CARDS_DIR = os.path.join(BASE_DIR, "cards")
PDFS_DIR = os.path.join(BASE_DIR, "pdfs")
PHOTOS_DIR = os.path.join(BASE_DIR, "photos")

os.makedirs(CARDS_DIR, exist_ok=True)
os.makedirs(PDFS_DIR, exist_ok=True)
os.makedirs(PHOTOS_DIR, exist_ok=True)

# -----------------------------
# --- Fonctions utilitaires ---
# -----------------------------
def _text_size(draw, text, font):
    bbox = draw.textbbox((0, 0), text, font=font)
    return bbox[2] - bbox[0], bbox[3] - bbox[1]

# -----------------------------
# --- Génération Carte ID ---
# -----------------------------
def generate_id_card(data):
    try:
        card_width, card_height = 860, 540
        img = Image.new("RGB", (card_width, card_height), (255, 255, 255))
        draw = ImageDraw.Draw(img)

        # Bandes en haut et bas
        draw.rectangle([0, 0, card_width, 60], fill=(0, 128, 0))
        draw.rectangle([0, card_height - 60, card_width, card_height], fill=(204, 0, 0))

        # Polices
        try:
            font_title = ImageFont.truetype("arialbd.ttf", 35)
            font_label = ImageFont.truetype("arial.ttf", 28)
            font_value = ImageFont.truetype("arialbd.ttf", 28)
            font_footer = ImageFont.truetype("ariali.ttf", 25)
        except:
            font_title = font_label = font_value = font_footer = ImageFont.load_default()

        # En-tête
        header_text = "ÉLÈVE – Redeeming Love Ministries"
        w, _ = _text_size(draw, header_text, font_title)
        draw.text(((card_width - w) / 2, 15), header_text, fill=(255, 255, 255), font=font_title)

        # Infos à gauche
        x_label = 50
        y_text = 90
        line_space = 34
        fields = [
            ("Nom :", data.get("nom", "")),
            ("Post-nom :", data.get("postnom", "")),
            ("Prénom :", data.get("prenom", "")),
            ("Matricule :", data.get("matricule", "")),
            ("Institution :", data.get("institution", "")),
            ("Classe :", data.get("classe", "")),
            ("Sexe :", data.get("sexe", "")),
            ("Naissance :", data.get("naissance", "")),
            ("Nom du Père :", data.get("pere", "")),
            ("Nom de la Mère :", data.get("mere", "")),
        ]
        for label, value in fields:
            draw.text((x_label, y_text), label, fill=(0, 0, 0), font=font_label)
            lw, _ = _text_size(draw, label, font_label)
            color = (0, 128, 0) if label.startswith("Matricule") else (0, 0, 0)
            draw.text((x_label + lw + 10, y_text), value, fill=color, font=font_value)
            y_text += line_space

        # Photo à droite
        photo_path = data.get("photo", "")
        photo_x, photo_y = card_width - 270, 90
        if os.path.exists(photo_path):
            try:
                photo = Image.open(photo_path).convert("RGB")
                photo.thumbnail((220, 280), Image.Resampling.LANCZOS)
                photo_x = card_width - photo.width - 50
                img.paste(photo, (photo_x, photo_y))
            except Exception as e:
                print(f"⚠️ Erreur chargement photo : {e}")
        else:
            print(f"⚠️ Photo introuvable : {photo_path}")

        # QR Code sous la photo
        qr_data = f"{data.get('matricule', '')} | {data.get('nom', '')} {data.get('postnom', '')} {data.get('prenom', '')}"
        qr = qrcode.QRCode(version=1, box_size=6, border=1)
        qr.add_data(qr_data)
        qr.make(fit=True)
        qr_img = qr.make_image(fill_color="black", back_color="white").convert("RGB")
        qr_img = qr_img.resize((140, 140), Image.Resampling.LANCZOS)
        qr_x = photo_x + (220 - 140) // 2
        qr_y = photo_y + 280 + 10
        img.paste(qr_img, (qr_x, qr_y))

        # Pied de page
        footer_text = f"Fait à Bukavu, le {datetime.now().strftime('%d/%m/%Y')}"
        fw, _ = _text_size(draw, footer_text, font_footer)
        draw.text(((card_width - fw) / 2, card_height - 45), footer_text, fill=(255, 255, 255), font=font_footer)

        # Sauvegarde
        safe_name = f"{data.get('nom', '')}_{data.get('prenom', '')}_{data.get('matricule', '')}".replace(" ", "_")
        card_path = os.path.join(CARDS_DIR, f"{safe_name}.png")
        img.save(card_path)
        print(f"✅ Carte enregistrée : {card_path}")
        return card_path

    except Exception as e:
        print(f"❌ Erreur génération carte : {e}")
        return None

# -----------------------------
# --- Génération PDF ---
# -----------------------------
def generate_pdf(data):
    try:
        safe_name = f"{data.get('nom','')}_{data.get('prenom','')}_{data.get('matricule','')}".replace(" ", "_")
        pdf_path = os.path.join(PDFS_DIR, f"{safe_name}.pdf")
        c = canvas.Canvas(pdf_path, pagesize=A4)

        c.setFont("Helvetica-Bold", 16)
        c.drawCentredString(105 * mm, 280 * mm, "FICHE D'IDENTIFICATION DE L'ÉLÈVE")
        c.setFont("Helvetica", 11)

        infos = [
            f"Matricule : {data.get('matricule','')}",
            f"Nom : {data.get('nom','')}",
            f"Post-nom : {data.get('postnom','')}",
            f"Prénom : {data.get('prenom','')}",
            f"Date de naissance : {data.get('naissance','')}",
            f"Sexe : {data.get('sexe','')}",
            f"Institution : {data.get('institution','')}",
            f"Classe : {data.get('classe','')}",
            f"Nom du Père : {data.get('pere','')}",
            f"Nom de la Mère : {data.get('mere','')}",
        ]

        y = 260 * mm
        for info in infos:
            c.drawString(25 * mm, y, info)
            y -= 8 * mm

        # Ajouter carte ID si existante
        card_path = os.path.join(CARDS_DIR, f"{safe_name}.png")
        if os.path.exists(card_path):
            c.drawImage(card_path, 25 * mm, 90 * mm, width=150 * mm, preserveAspectRatio=True)
        else:
            print(f"⚠️ Carte non trouvée pour inclusion PDF : {card_path}")

        c.setFont("Helvetica-Oblique", 10)
        c.drawCentredString(105 * mm, 20 * mm, f"Fait à Bukavu, le {datetime.now().strftime('%d/%m/%Y')}")

        c.save()
        print(f"✅ PDF généré : {pdf_path}")
        return pdf_path

    except Exception as e:
        print(f"❌ Erreur génération PDF : {e}")
        return None

# -----------------------------
# --- MAIN pour test ---
# -----------------------------
if __name__ == "__main__":
    # --- Définir gen_data avant d'appeler les fonctions ---
    gen_data = {
        "nom": "Kabasele",
        "postnom": "Mbuyi",
        "prenom": "Jean",
        "matricule": "2025-001",
        "institution": "RL Ministries",
        "classe": "6ème",
        "sexe": "M",
        "naissance": "12/03/2010",
        "pere": "Patrick Kabasele",
        "mere": "Marie Mbuyi",
        "photo": os.path.join(PHOTOS_DIR, "jean.jpg")  # mettre une vraie photo ici
    }

    # --- Génération carte + PDF ---
    card_path = generate_id_card(gen_data)
    pdf_path = generate_pdf(gen_data)
