# id_card.py
import os
from PIL import Image, ImageDraw, ImageFont
import qrcode

RESOURCES_DIR = os.path.join(os.getcwd(), "resources")
LOGO_PATH = os.path.join(RESOURCES_DIR, "logo.png")
FONT_PATH = os.path.join(RESOURCES_DIR, "Arial.ttf")  # Assurez-vous d'avoir Arial.ttf ou autre police

def generate_qr_code(data: str, size=150):
    qr = qrcode.QRCode(
        version=1,
        error_correction=qrcode.constants.ERROR_CORRECT_H,
        box_size=3,
        border=2,
    )
    qr.add_data(data)
    qr.make(fit=True)
    img = qr.make_image(fill_color="black", back_color="white")
    return img.resize((size, size))

def generate_id_card(data: dict, output_path: str, card_width=600, card_height=350):
    """
    Génère un ID CARD PNG avec logo, couleur standard (vert/rouge) et QR code.
    data: dict contenant les infos de l'étudiant/famille/nutrition
    output_path: chemin du fichier PNG à générer
    """
    # Couleurs
    bg_color = (0, 128, 0)  # vert de fond
    accent_color = (200, 0, 0)  # rouge accent

    card = Image.new("RGB", (card_width, card_height), color=bg_color)
    draw = ImageDraw.Draw(card)

    # Charger logo
    if os.path.exists(LOGO_PATH):
        logo = Image.open(LOGO_PATH).convert("RGBA")
        logo.thumbnail((100, 100))
        card.paste(logo, (20, 20), logo)

    # Charger photo de l’utilisateur
    photo_path = data.get("photo_path")
    if photo_path and os.path.exists(photo_path):
        photo = Image.open(photo_path).convert("RGBA")
        photo.thumbnail((120, 120))
        card.paste(photo, (card_width - 140, 20), photo)

    # QR Code (matricule)
    qr_img = generate_qr_code(data.get("matricule", ""))
    card.paste(qr_img, (card_width - 140, card_height - 140))

    # Texte
    font = ImageFont.truetype(FONT_PATH, 18) if os.path.exists(FONT_PATH) else ImageFont.load_default()

    x_text = 20
    y_text = 140
    line_height = 28

    info_lines = [
        f"Matricule: {data.get('matricule','')}",
        f"Nom: {data.get('nom','')} {data.get('post_nom','')}",
        f"Prénom: {data.get('prenom','')}",
        f"Père: {data.get('pere','')}",
        f"Mère: {data.get('mere','')}",
        f"Naissance: {data.get('naissance','')}",
        f"Sexe: {data.get('sexe','')}"
    ]

    for line in info_lines:
        draw.text((x_text, y_text), line, fill="white", font=font)
        y_text += line_height

    # Encadré rouge en bas
    draw.rectangle([(0, card_height-40), (card_width, card_height)], fill=accent_color)

    # Sauvegarder
    card.save(output_path)
