# gui/students_page.py
import os
import cv2
import sqlite3
import time
from datetime import datetime
from PyQt6.QtWidgets import (
    QWidget, QLabel, QLineEdit, QPushButton, QVBoxLayout, QHBoxLayout,
    QFormLayout, QMessageBox, QComboBox, QSizePolicy,
    QTableWidget, QTableWidgetItem, QDialog
)
from PyQt6.QtGui import QPixmap, QImage, QFont
from PyQt6.QtCore import Qt, QTimer
from id_card_pdf import generate_id_card, generate_pdf  # Génération ID & PDF
import glob

# ---------------------------
# Fichiers et dossiers
# ---------------------------
BASE_DIR = os.getcwd()
DB_DIR = os.path.join(BASE_DIR, "database")
STUDENTS_DB = os.path.join(DB_DIR, "students.db")
CARDS_DIR = os.path.join(BASE_DIR, "cards")
PDFS_DIR = os.path.join(BASE_DIR, "pdfs")

os.makedirs(DB_DIR, exist_ok=True)
os.makedirs(CARDS_DIR, exist_ok=True)
os.makedirs(PDFS_DIR, exist_ok=True)

# ---------------------------
# Vérification / creation de la table students et de ses colonnes
# ---------------------------
def ensure_students_table():
    """
    Crée la table students si elle n'existe pas et ajoute les colonnes manquantes.
    Colonnes attendues :
      matricule, nom, postnom, prenom, pere, mere, naissance, sexe,
      institution, classe, resultat, mention, photo, statut, owner, date_enreg
    """
    os.makedirs(os.path.dirname(STUDENTS_DB), exist_ok=True)
    conn = sqlite3.connect(STUDENTS_DB)
    c = conn.cursor()

    # Déclaration de la table avec toutes les colonnes attendues
    c.execute("""
        CREATE TABLE IF NOT EXISTS students (
            matricule TEXT PRIMARY KEY,
            nom TEXT,
            postnom TEXT,
            prenom TEXT,
            pere TEXT,
            mere TEXT,
            naissance TEXT,
            sexe TEXT,
            institution TEXT,
            classe TEXT,
            resultat TEXT,
            mention TEXT,
            photo TEXT,
            statut TEXT DEFAULT 'actif',
            owner TEXT,
            date_enreg TEXT DEFAULT CURRENT_TIMESTAMP
        )
    """)

    # Liste des colonnes attendues et leur type (pour ajout si manquent)
    columns = [
        ("matricule", "TEXT PRIMARY KEY"),
        ("nom", "TEXT"),
        ("postnom", "TEXT"),
        ("prenom", "TEXT"),
        ("pere", "TEXT"),
        ("mere", "TEXT"),
        ("naissance", "TEXT"),
        ("sexe", "TEXT"),
        ("institution", "TEXT"),
        ("classe", "TEXT"),
        ("resultat", "TEXT"),
        ("mention", "TEXT"),
        ("photo", "TEXT"),
        ("statut", "TEXT DEFAULT 'actif'"),
        ("owner", "TEXT"),
        ("date_enreg", "TEXT DEFAULT CURRENT_TIMESTAMP")
    ]

    # Vérifier et ajouter colonnes manquantes
    c.execute("PRAGMA table_info(students)")
    existing_cols = [col[1] for col in c.fetchall()]
    for col_name, col_type in columns:
        if col_name not in existing_cols:
            try:
                # On n'ajoute que la colonne (sqlite ignore PRIMARY KEY sur ALTER)
                # si col_type contient DEFAULT, on l'ajoute tel quel (possible)
                c.execute(f"ALTER TABLE students ADD COLUMN {col_name} {col_type}")
            except Exception as e:
                # Si échec, écrire en console pour debug (ne pas interrompre)
                print(f"⚠️ Impossible d’ajouter la colonne {col_name}: {e}")

    conn.commit()
    conn.close()

# Appel d'initialisation
ensure_students_table()

# ---------------------------
# Classe StudentsPage
# ---------------------------
class StudentsPage(QWidget):
    def __init__(self, parent=None, user_role="user", nom_utilisateur_actif="unknown"):
        super().__init__(parent)
        self.user_role = user_role
        self.nom_utilisateur_actif = nom_utilisateur_actif

        # Caméra
        self.cap = None
        self.timer = QTimer(self)
        self.timer.timeout.connect(self._query_frame)
        self.current_frame = None
        self.last_photo_path = None

        # Style
        self.setWindowTitle(f"Identification - CONNECTÉ : {self.nom_utilisateur_actif}")
        self.setStyleSheet("""
            QWidget {
                background-color: #999999;
                color: #000000;
                font-family: Arial;
                font-size: 12pt;
            }
            QLineEdit, QComboBox {
                background-color: #ffffff;
                border: 1px solid #555;
                padding: 4px;
                color: #000000;
            }
            QPushButton#saveBtn {
                background-color: #000066;
                border-radius: 5px;
                padding: 6px 12px;
                color: #fff;
            }
            QPushButton#clearBtn {
                background-color: #aa0000;
                border-radius: 5px;
                padding: 6px 12px;
                color: #fff;
            }
            QPushButton:hover {
                background-color: #777;
            }
            QLabel#titleLabel {
                font-size: 22pt;
                font-weight: bold;
                color: #000000;
            }
            QLabel#subtitleLabel {
                font-size: 18pt;
                font-weight: bold;
                color: #00ccff;
            }
            QTableWidget {
                background-color: #333333;
                border: 1px solid #555;
                color: #cccccc;
                gridline-color: #555;
                selection-background-color: #66cccc;
            }
            QHeaderView::section {
                background-color: #5a5a5a;
                color: #fff;
                padding: 4px;
                border: 1px solid #444;
            }
        """)

        # Layout principal vertical
        main_layout = QVBoxLayout(self)
        main_layout.setContentsMargins(12, 12, 12, 12)
        main_layout.setSpacing(10)

        # TITRE HAUT
        title_label = QLabel(f"IDENTIFICATION ÉLÈVE — CONNECTÉ : {self.nom_utilisateur_actif}")
        title_label.setObjectName("titleLabel")
        title_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        main_layout.addWidget(title_label)

        # Layout horizontal : formulaire + photo
        content_layout = QHBoxLayout()
        content_layout.setSpacing(12)

        # Formulaire
        form_layout = QFormLayout()
        form_layout.setLabelAlignment(Qt.AlignmentFlag.AlignRight)
        form_layout.setFormAlignment(Qt.AlignmentFlag.AlignLeft | Qt.AlignmentFlag.AlignTop)

        self.input_matricule = QLineEdit()
        self.input_nom = QLineEdit()
        self.input_postnom = QLineEdit()
        self.input_prenom = QLineEdit()
        self.input_pere = QLineEdit()
        self.input_mere = QLineEdit()
        self.input_naissance = QLineEdit()
        self.input_sexe = QComboBox()
        self.input_sexe.addItems(["", "Masculin", "Féminin"])
        self.input_institution = QLineEdit()
        self.input_classe = QLineEdit()
        self.input_resultat = QLineEdit()
        self.input_mention = QComboBox()
        self.input_mention.addItems(["", "Accepté(e)", "Refusé(e)", "Réévalué(e)"])

        for label, widget in [
            ("Matricule:", self.input_matricule),
            ("Nom:", self.input_nom),
            ("Post-Nom:", self.input_postnom),
            ("Prénom:", self.input_prenom),
            ("Nom du Père:", self.input_pere),
            ("Nom de la Mère:", self.input_mere),
            ("Date Naissance:", self.input_naissance),
            ("Sexe:", self.input_sexe),
            ("Institution:", self.input_institution),
            ("Classe:", self.input_classe),
            ("Résultat:", self.input_resultat),
            ("Mention:", self.input_mention)
        ]:
            form_layout.addRow(label, widget)

        actions_layout = QHBoxLayout()
        self.btn_save = QPushButton("Enregistrer")
        self.btn_save.setObjectName("saveBtn")
        self.btn_clear = QPushButton("Effacer")
        self.btn_clear.setObjectName("clearBtn")
        actions_layout.addWidget(self.btn_save)
        actions_layout.addWidget(self.btn_clear)
        form_layout.addRow(actions_layout)

        left_widget = QWidget()
        left_widget.setLayout(form_layout)
        left_widget.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Expanding)
        content_layout.addWidget(left_widget, stretch=2)

        # Photo et caméra
        right_layout = QVBoxLayout()
        right_layout.setSpacing(8)

        self.photo_label = QLabel("Aucune photo")
        self.photo_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.photo_label.setFixedSize(210, 260)
        self.photo_label.setStyleSheet(
            "QLabel { border: 2px dashed #999; background: #1e1e1e; color: white; }"
        )
        right_layout.addWidget(self.photo_label, alignment=Qt.AlignmentFlag.AlignHCenter)

        # Boutons caméra
        camera_btns = QHBoxLayout()
        self.btn_open_camera = QPushButton("Ouvrir Caméra")
        self.btn_capture = QPushButton("Capturer Photo")
        self.btn_close_camera = QPushButton("Fermer Caméra")
        self.btn_capture.setEnabled(False)
        self.btn_close_camera.setEnabled(False)
        camera_btns.addWidget(self.btn_open_camera)
        camera_btns.addWidget(self.btn_capture)
        camera_btns.addWidget(self.btn_close_camera)
        right_layout.addLayout(camera_btns)

        # Barre de recherche directement sous les boutons de capture
        self.search_input = QLineEdit()
        self.search_input.setPlaceholderText("Rechercher par matricule ou nom...")
        self.search_input.setStyleSheet("""
            QLineEdit {
                background-color: #005556;
                border: 2px solid #0055aa;
                border-radius: 5px;
                padding: 4px;
                color: #ffffff;
            }
            QLineEdit:focus {
                border: 2px solid #00aaff;
            }
        """)
        right_layout.addWidget(self.search_input)

        self.info_label = QLabel("")
        self.info_label.setWordWrap(True)
        right_layout.addWidget(self.info_label)

        content_layout.addLayout(right_layout, stretch=1)
        main_layout.addLayout(content_layout)

        # Tableau des étudiants
        self.students_table = QTableWidget()
        self.students_table.setColumnCount(12)
        self.students_table.setHorizontalHeaderLabels([
            "Matricule", "Nom", "Postnom", "Prénom", "Père", "Mère",
            "Naissance", "Sexe", "Institution", "Classe", "Résultat", "Mention"
        ])
        self.students_table.horizontalHeader().setStretchLastSection(True)
        self.students_table.setSelectionBehavior(QTableWidget.SelectionBehavior.SelectRows)
        self.students_table.setEditTriggers(QTableWidget.EditTrigger.NoEditTriggers)
        self.students_table.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Expanding)
        main_layout.addWidget(QLabel("DB Étudiants :"))
        main_layout.addWidget(self.students_table)

        # TITRE BAS
        subtitle_label = QLabel("Identification de l'Eleve")
        subtitle_label.setObjectName("subtitleLabel")
        subtitle_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        main_layout.addWidget(subtitle_label)
        main_layout.addStretch()

        # Connexions (inchangées)
        self.btn_open_camera.clicked.connect(self.start_camera)
        self.btn_capture.clicked.connect(self.capture_photo)
        self.btn_close_camera.clicked.connect(self.stop_camera)
        self.btn_clear.clicked.connect(self.clear_form)
        self.btn_save.clicked.connect(self.save_student)
        self.search_input.textChanged.connect(self.search_students)
        self.students_table.cellDoubleClicked.connect(self.show_id_card)

        self.update_students_list()

    # ===========================
    # Fonctions Caméra et capture
    # ===========================
    def start_camera(self):
        if self.cap is not None:
            return
        self.cap = cv2.VideoCapture(0, cv2.CAP_DSHOW)
        if not self.cap.isOpened():
            try:
                self.cap.release()
            except Exception:
                pass
            self.cap = cv2.VideoCapture(0)
        if not self.cap.isOpened():
            QMessageBox.warning(self, "Caméra", "Impossible d'ouvrir la caméra.")
            self.cap = None
            return
        self.timer.start(30)
        self.btn_open_camera.setEnabled(False)
        self.btn_capture.setEnabled(True)
        self.btn_close_camera.setEnabled(True)
        self.info_label.setText("Caméra ouverte. Cliquez sur Capturer.")

    def _query_frame(self):
        if not self.cap:
            return
        ret, frame = self.cap.read()
        if not ret:
            return
        self.current_frame = frame
        try:
            frame_rgb = cv2.cvtColor(frame, cv2.COLOR_BGR2RGB)
            h, w, ch = frame_rgb.shape
            bytes_per_line = ch * w
            qimg = QImage(frame_rgb.data, w, h, bytes_per_line, QImage.Format.Format_RGB888)
            pix = QPixmap.fromImage(qimg)
            pix = pix.scaled(self.photo_label.width(),
                             self.photo_label.height(),
                             Qt.AspectRatioMode.KeepAspectRatioByExpanding,
                             Qt.TransformationMode.SmoothTransformation)
            self.photo_label.setPixmap(pix)
        except Exception:
            pass

    def capture_photo(self):
        if self.current_frame is None:
            self.info_label.setText("Aucune image à capturer.")
            return
        try:
            self.timer.stop()
            frame_height, frame_width, _ = self.current_frame.shape
            label_width = self.photo_label.width()
            label_height = self.photo_label.height()

            frame_ratio = frame_width / frame_height
            label_ratio = label_width / label_height

            if frame_ratio > label_ratio:
                new_width = int(frame_height * label_ratio)
                x_start = (frame_width - new_width) // 2
                cropped = self.current_frame[:, x_start:x_start + new_width]
            else:
                new_height = int(frame_width / label_ratio)
                y_start = (frame_height - new_height) // 2
                cropped = self.current_frame[y_start:y_start + new_height, :]

            display_img = cv2.resize(cropped, (label_width, label_height), interpolation=cv2.INTER_LANCZOS4)

            os.makedirs(CARDS_DIR, exist_ok=True)
            photo_filename = f"student_{int(time.time())}.jpg"
            photo_path = os.path.join(CARDS_DIR, photo_filename)
            cv2.imwrite(photo_path, cropped)

            self.last_photo_path = photo_path
            self.info_label.setText(f"Photo capturée et enregistrée : {photo_filename}")

            frame_rgb = cv2.cvtColor(display_img, cv2.COLOR_BGR2RGB)
            h, w, ch = frame_rgb.shape
            bytes_per_line = ch * w
            qimg = QImage(frame_rgb.data, w, h, bytes_per_line, QImage.Format.Format_RGB888)
            pix = QPixmap.fromImage(qimg)
            self.photo_label.setPixmap(pix)

            self.btn_capture.setEnabled(False)
            self.btn_open_camera.setEnabled(True)
        except Exception as e:
            self.info_label.setText(f"Erreur lors de la capture : {e}")

    def stop_camera(self):
        if self.timer.isActive():
            self.timer.stop()
        if self.cap:
            try:
                self.cap.release()
            except Exception:
                pass
            self.cap = None
        self.photo_label.setText("Aucune photo")
        self.photo_label.setPixmap(QPixmap())
        self.btn_open_camera.setEnabled(True)
        self.btn_capture.setEnabled(False)
        self.btn_close_camera.setEnabled(False)
        self.info_label.setText("Caméra fermée.")

    # ===========================
    # Form utilities
    # ===========================
    def clear_form(self):
        self.input_matricule.clear()
        self.input_nom.clear()
        self.input_postnom.clear()
        self.input_prenom.clear()
        self.input_pere.clear()
        self.input_mere.clear()
        self.input_naissance.clear()
        self.input_sexe.setCurrentIndex(0)
        self.input_institution.clear()
        self.input_classe.clear()
        self.input_resultat.clear()
        self.input_mention.setCurrentIndex(0)
        self.photo_label.setText("Aucune photo")
        self.photo_label.setPixmap(QPixmap())
        self.last_photo_path = None
        self.info_label.setText("Formulaire effacé.")

    def save_student(self):
        # Exige photo
        if not self.last_photo_path:
            QMessageBox.warning(self, "Erreur", "Capturez une photo avant d'enregistrer.")
            return

        # Définir owner depuis le compte actif
        current_owner = getattr(self, "nom_utilisateur_actif", "unknown")

        # Construire le dictionnaire des données
        data = {
            "matricule": self.input_matricule.text().strip() or None,
            "nom": self.input_nom.text().strip() or None,
            "postnom": self.input_postnom.text().strip() or None,
            "prenom": self.input_prenom.text().strip() or None,
            "pere": self.input_pere.text().strip() or None,
            "mere": self.input_mere.text().strip() or None,
            "naissance": self.input_naissance.text().strip() or None,
            "sexe": self.input_sexe.currentText() or None,
            "institution": self.input_institution.text().strip() or None,
            "classe": self.input_classe.text().strip() or None,
            "resultat": self.input_resultat.text().strip() or None,
            "mention": self.input_mention.currentText() or None,
            "photo": self.last_photo_path,
            "statut": "actif",
            "owner": current_owner,
            "date_enreg": datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        }

        try:
            conn = sqlite3.connect(STUDENTS_DB)
            c = conn.cursor()

            # Récupération des colonnes réelles de la table
            c.execute("PRAGMA table_info(students)")
            columns_info = c.fetchall()
            columns = [col[1] for col in columns_info]
            num_columns = len(columns)

            # Construire la liste de valeurs dans le même ordre que les colonnes
            values = [data.get(col, None) for col in columns]

            # Construire requête dynamique
            placeholders = ",".join(["?"] * num_columns)
            query = f"INSERT OR REPLACE INTO students VALUES ({placeholders})"
            c.execute(query, tuple(values))
            conn.commit()
            conn.close()

            # Génération de la carte et du PDF
            try:
                card_path = generate_id_card(data)
            except Exception as e:
                print(f"⚠️ Erreur génération carte : {e}")
            try:
                pdf_path = generate_pdf(data)
            except Exception as e:
                print(f"⚠️ Erreur génération PDF : {e}")

            QMessageBox.information(self, "Succès",
                                    "Étudiant enregistré avec succès.\nCarte et PDF générés.")
            self.clear_form()
            self.update_students_list()

        except Exception as e:
            QMessageBox.warning(self, "Erreur",
                                f"Impossible d'enregistrer complètement :\n{e}")

    # ===========================
    # Liste & recherche étudiants
    # ===========================
    def update_students_list(self):
        self.students_table.setRowCount(0)
        conn = sqlite3.connect(STUDENTS_DB)
        c = conn.cursor()

        # Filtrer selon l'utilisateur actif (owner)
        current_owner = getattr(self, "nom_utilisateur_actif", "unknown")

        try:
            # Si admin -> voir tous
            if self.user_role and str(self.user_role).lower() == "admin":
                c.execute("""
                    SELECT matricule, nom, postnom, prenom, pere, mere, naissance, sexe,
                           institution, classe, resultat, mention
                    FROM students
                    ORDER BY rowid DESC
                """)
            else:
                # S'assurer que la colonne owner existe, sinon fallback pour ne pas planter
                c.execute("PRAGMA table_info(students)")
                cols = [col[1] for col in c.fetchall()]
                if "owner" in cols:
                    c.execute("""
                        SELECT matricule, nom, postnom, prenom, pere, mere, naissance, sexe,
                               institution, classe, resultat, mention
                        FROM students
                        WHERE owner = ?
                        ORDER BY rowid DESC
                    """, (current_owner,))
                else:
                    # fallback: afficher tout si pas de owner
                    c.execute("""
                        SELECT matricule, nom, postnom, prenom, pere, mere, naissance, sexe,
                               institution, classe, resultat, mention
                        FROM students
                        ORDER BY rowid DESC
                    """)

            rows = c.fetchall()
            for row_idx, row_data in enumerate(rows):
                self.students_table.insertRow(row_idx)
                for col_idx, value in enumerate(row_data):
                    item = QTableWidgetItem(str(value))
                    item.setFlags(Qt.ItemFlag.ItemIsEnabled | Qt.ItemFlag.ItemIsSelectable)
                    self.students_table.setItem(row_idx, col_idx, item)
        except Exception as e:
            QMessageBox.warning(self, "Erreur DB", f"Impossible de charger students: {e}")
        finally:
            conn.close()

    def search_students(self, text):
        text = text.lower()
        for row in range(self.students_table.rowCount()):
            match = False
            for col in range(self.students_table.columnCount()):
                item = self.students_table.item(row, col)
                if item is not None and text in item.text().lower():
                    match = True
                    break
            self.students_table.setRowHidden(row, not match)

    def show_id_card(self, row, column):
        # Surligner la ligne sélectionnée
        self.students_table.selectRow(row)

        matricule_item = self.students_table.item(row, 0)
        nom_item = self.students_table.item(row, 1)
        prenom_item = self.students_table.item(row, 3)

        if not matricule_item or not nom_item or not prenom_item:
            QMessageBox.warning(self, "Erreur", "Impossible de récupérer les informations de l'étudiant.")
            return

        matricule = matricule_item.text()
        nom = nom_item.text().upper()
        prenom = prenom_item.text().upper()

        # Construire le pattern du fichier
        pattern = f"{nom}_{prenom}_{matricule}.png"

        # Chercher uniquement dans le dossier des cartes
        card_path = os.path.join(CARDS_DIR, pattern)
        if not os.path.exists(card_path):
            QMessageBox.warning(self, "Carte introuvable", f"Aucune carte trouvée pour {nom} {prenom} ({matricule}) dans le dossier {CARDS_DIR}.")
            return

        # Afficher la carte dans un QDialog
        dlg = QDialog(self)
        dlg.setWindowTitle(f"Carte d'identité - {nom} {prenom}")
        dlg.setModal(True)
        layout = QVBoxLayout(dlg)

        label = QLabel()
        pix = QPixmap(card_path)
        pix = pix.scaled(400, 500, Qt.AspectRatioMode.KeepAspectRatio, Qt.TransformationMode.SmoothTransformation)
        label.setPixmap(pix)
        label.setAlignment(Qt.AlignmentFlag.AlignCenter)

        layout.addWidget(label)
        dlg.setLayout(layout)
        dlg.exec()
if __name__=="__main__":
    import sys
    app=QApplication(sys.argv)
    window=StudentsPage(user_role="admin", nom_utilisateur_actif="TestUser")
    sys.exit(app.exec())