# gui/student_panel.py
import os
import sys
import sqlite3
import time
import re
import shutil
import cv2
from pyzbar.pyzbar import decode
from PyQt6.QtWidgets import (
    QApplication, QWidget, QHBoxLayout, QVBoxLayout, QPushButton,
    QLabel, QLineEdit, QTableWidget, QTableWidgetItem,
    QMessageBox, QSizePolicy, QInputDialog, QFrame, QFileDialog
)
from PyQt6.QtGui import QFont, QPixmap, QImage
from PyQt6.QtCore import Qt, QTimer
from id_card_pdf import generate_id_card, generate_pdf  # ton module existant
from gui.students_page import StudentsPage  # page pour nouvelle identification
import pandas as pd
from PIL import Image
from datetime import datetime
#from gui.login_page import LoginPage
from PyQt6.QtWidgets import QMainWindow, QMessageBox
from PyQt6.QtWidgets import (
    QApplication, QWidget, QVBoxLayout, QHBoxLayout, QLabel,
    QPushButton, QMessageBox, QFileDialog, QInputDialog, QProgressDialog
)
PHOTOS_DIR = r"D:\dlogi_app\PhotoCarte"
BASE_DIR = os.getcwd()
DB_DIR = os.path.join(BASE_DIR, "database")
STUDENTS_DB = os.path.join(DB_DIR, "students.db")
USERS_DB = os.path.join(DB_DIR, "users.db")
CARDS_DIR = os.path.join(BASE_DIR, "cards")
PDFS_DIR = os.path.join(BASE_DIR, "pdfs")
LOGO_PATH = os.path.join(BASE_DIR, "logo.png")
PROFILE_PHOTOS_DIR = os.path.join(DB_DIR, "profile_photos")

os.makedirs(DB_DIR, exist_ok=True)
os.makedirs(CARDS_DIR, exist_ok=True)
os.makedirs(PDFS_DIR, exist_ok=True)
os.makedirs(PROFILE_PHOTOS_DIR, exist_ok=True)

BASE_DIR = os.path.dirname(os.path.abspath(__file__))  # répertoire de l'application
PHOTOS_DIR = os.path.join(BASE_DIR, "photos")          # dossier photos
if not os.path.exists(PHOTOS_DIR):
    os.makedirs(PHOTOS_DIR)

PHOTOS_DIR = os.path.join(os.getcwd(), "PhotoCarte")  # dossier local PhotoCarte
os.makedirs(PHOTOS_DIR, exist_ok=True)

# ---------------------------
# Initialisation DB si besoin
# ---------------------------
def init_students_db():
    conn = sqlite3.connect(STUDENTS_DB)
    c = conn.cursor()
    # table students (avec owner)
    c.execute("""
        CREATE TABLE IF NOT EXISTS students (
            matricule TEXT PRIMARY KEY,
            nom TEXT, postnom TEXT, prenom TEXT,
            pere TEXT, mere TEXT, naissance TEXT,
            sexe TEXT, institution TEXT, classe TEXT,
            resultat TEXT, mention TEXT, photo TEXT,
            owner TEXT DEFAULT 'admin'
        )
    """)
    # table access_qr (avec owner)
    c.execute("""
        CREATE TABLE IF NOT EXISTS access_qr (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            matricule TEXT,
            nom TEXT,
            postnom TEXT,
            prenom TEXT,
            timestamp TEXT,
            owner TEXT DEFAULT 'admin'
        )
    """)
    # families & nutrition minimal (so we can show stats)
    c.execute("""
        CREATE TABLE IF NOT EXISTS families (
            matricule TEXT PRIMARY KEY,
            nom_famille TEXT,
            responsable TEXT,
            zone TEXT,
            adresse TEXT,
            nb_enfants INTEGER,
            categorie TEXT,
            date_debut_suivi TEXT,
            date_fin_suivi TEXT,
            photo TEXT,
            owner TEXT DEFAULT 'admin',
            date_enreg TEXT DEFAULT CURRENT_TIMESTAMP
        )
    """)
    c.execute("""
        CREATE TABLE IF NOT EXISTS nutrition (
            num_dossier TEXT PRIMARY KEY,
            nom TEXT,
            sexe TEXT,
            date_naissance TEXT,
            nom_parent TEXT,
            adresse TEXT,
            date_enreg TEXT,
            poids REAL,
            pb REAL,
            scorez REAL,
            oedemes TEXT,
            appetit_bon TEXT,
            autres TEXT,
            photo TEXT
        )
    """)
    conn.commit()
    conn.close()

def ensure_owner_columns():
    """
    Au cas où la DB préexistante n'a pas les colonnes owner,
    on tente de les ajouter sans casser l'existant.
    """
    conn = sqlite3.connect(STUDENTS_DB)
    c = conn.cursor()
    try:
        c.execute("PRAGMA table_info(students)")
        cols_students = [x[1] for x in c.fetchall()]
        if "owner" not in cols_students:
            try:
                c.execute("ALTER TABLE students ADD COLUMN owner TEXT DEFAULT 'admin'")
            except Exception:
                pass

        c.execute("PRAGMA table_info(access_qr)")
        cols_access = [x[1] for x in c.fetchall()]
        if "owner" not in cols_access:
            try:
                c.execute("ALTER TABLE access_qr ADD COLUMN owner TEXT DEFAULT 'admin'")
            except Exception:
                pass
        conn.commit()
    except Exception as e:
        print(f"Erreur ajout colonne owner: {e}")
    finally:
        conn.close()

def ensure_users_table():
    conn = sqlite3.connect(USERS_DB)
    c = conn.cursor()
    c.execute("""
        CREATE TABLE IF NOT EXISTS users (
            username TEXT PRIMARY KEY,
            password TEXT,
            role TEXT DEFAULT 'user',
            profile_photo TEXT
        )
    """)
    conn.commit()
    conn.close()

init_students_db()
ensure_owner_columns()
ensure_users_table()

# ---------------------------
# Helpers: photo crop/resize (optionnel, tu voulais l'intégrer)
# ---------------------------
def crop_and_resize_photo(photo_path, target_size=(300, 400)):
    """
    Rogne et redimensionne une photo au centre pour correspondre au format voulu.
    """
    try:
        img = Image.open(photo_path)
        width, height = img.size
        target_w, target_h = target_size

        aspect_ratio = width / height
        target_ratio = target_w / target_h

        if aspect_ratio > target_ratio:
            new_width = int(height * target_ratio)
            left = (width - new_width) // 2
            right = left + new_width
            img = img.crop((left, 0, right, height))
        else:
            new_height = int(width / target_ratio)
            top = (height - new_height) // 2
            bottom = top + new_height
            img = img.crop((0, top, width, bottom))

        img = img.resize(target_size, Image.Resampling.LANCZOS)
        img.save(photo_path)
        return photo_path
    except Exception as e:
        print(f"⚠️ Erreur rognage/redimensionnement photo {photo_path}: {e}")
        return photo_path

# ---------------------------
# Normalisation du texte scanné (POS/USB)
# ---------------------------
def normalize_scanned(qr_raw: str) -> str:
    """
    Prend la chaîne brute renvoyée par le scanner et renvoie une chaîne
    normalisée du type "ID|Nom Postnom Prénom".
    - conserve les '-' (ne les remplace pas)
    - si le séparateur '|' a été mal lu (ex: '~', '¦', ';', ':'...), on le détecte
      et on recrée la séparation correcte.
    - nettoie les caractères non lettre/chiffre/espaces/ '-' en espaces dans la partie nom.
    """
    if not qr_raw:
        return ""

    s = qr_raw.strip()

    # Normalisation de certains caractères d'espacement invisibles
    s = s.replace('\r', ' ').replace('\n', ' ').strip()

    # Si le séparateur vertical est déjà présent, on s'en sert rapidement (on garde tout après le premier '|')
    if "~" in s:
        parts = s.split("~", 1)
        id_part = parts[0].strip()
        name_part = parts[1].strip()
        # Nettoyage du name_part: garder lettres, chiffres, espaces, accents, et '-'
        name_part = re.sub(r"[^\w\s\-\u00C0-\u024F]", " ", name_part)
        name_part = re.sub(r"\s+", " ", name_part).strip()
        return f"{id_part}~{name_part}"

    # Si pas de '|' : essayer de repérer un séparateur parmi caractères souvent mal lus
    sep_candidates_pattern = r"[~¬¦¡;:/\\\u00A6\u00AC\u00B0\|\u2016\u201A\u201B\u201C\u201D\u2039\u203A]"  # liste de candidats
    m = re.search(sep_candidates_pattern, s)
    if m:
        idx = m.start()
        id_part = s[:idx].strip()
        name_part = s[idx+1:].strip()
        name_part = re.sub(r"[^\w\s\-\u00C0-\u024F]", " ", name_part)
        name_part = re.sub(r"\s+", " ", name_part).strip()
        return f"{id_part}~{name_part}"

    # Heuristique : si la chaîne commence par un ID (alphanumérique +/- '-') suivi d'un espace,
    # on le prend comme matricule
    m2 = re.match(r"\s*([A-Za-z0-9\-]+)\s+(.*)$", s, re.DOTALL)
    if m2:
        id_part = m2.group(1).strip()
        name_part = m2.group(2).strip()
        name_part = re.sub(r"[^\w\s\-\u00C0-\u024F]", " ", name_part)
        name_part = re.sub(r"\s+", " ", name_part).strip()
        return f"{id_part}|{name_part}" if name_part else f"{id_part}~"

    # fallback : remplacer groupes de symboles (sauf '-') par espace, prendre premier token comme id
    fallback = re.sub(r"[^\w\-\u00C0-\u024F]+", " ", s)
    fallback = re.sub(r"\s+", " ", fallback).strip()
    tokens = fallback.split(" ", 1)
    if tokens:
        id_token = tokens[0]
        name_rest = tokens[1] if len(tokens) > 1 else ""
        return f"{id_token}~{name_rest.strip()}"
    return s

# ---------------------------
# StudentPanel
# ---------------------------
class StudentPanel(QWidget):
    def __init__(self, user_role="user", nom_utilisateur_actif="Admin", admin_password="admin123", username="Admin"):
        super().__init__()
        self.user_role = user_role
        self.nom_utilisateur_actif = nom_utilisateur_actif
        self.admin_password = admin_password
        self.username = username  # ✅ stocke le nom d'utilisateur correctement

        # UI state
        self.qr_count_label = QLabel("Cartes scannées : 0")
        self._qr_tab_initialized = False  # évite recréation multiple de widgets QR
        self.profile_photo_path = None

        self.setWindowTitle("Student Panel - RLM")
        self.setMinimumSize(1200, 700)
        self.setStyleSheet("""
            QWidget {background-color:#1e1e1e;color:#f0f0f0;font-family:'Segoe UI';}
            QPushButton {background-color:#2d2d2d;color:#f0f0f0;padding:8px;border-radius:6px;}
            QPushButton:hover {background-color:#3a3a3a;}
            QLineEdit {padding:6px;border-radius:6px;background:#2b2b2b;color:#f0f0f0;}
            QLabel#sidebarTitle { font-weight: bold; color: #fff; }
            QHeaderView::section {background-color:#444;color:white;padding:6px;border:1px solid #555;}
            QTableWidget {gridline-color:#555;background-color:#999999;alternate-background-color:#2d2d2d;color:#333333;}
            QFrame#cameraFrame { border: 2px solid #0055aa; background-color: #000; }
        """)

        # Caméra
        self.cap = None
        self.camera_timer = QTimer(self)
        self.camera_timer.timeout.connect(self._camera_grab)
        self.camera_running = False

        # Layout principal
        main_layout = QHBoxLayout(self)
        menu_layout = QVBoxLayout()
        menu_layout.setSpacing(8)

        # Titre et sous-titre (titre stocké pour mise à jour ultérieure)
        self.title_label = QLabel(f"CONNECTED : RLM / DRC {self.nom_utilisateur_actif}")
        self.title_label.setObjectName("sidebarTitle")
        self.title_label.setFont(QFont("Segoe UI", 14, QFont.Weight.Bold))
        self.title_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        menu_layout.addWidget(self.title_label)

        subtitle = QLabel("RLM - ID-Logi builder____D-Tech")
        subtitle.setFont(QFont("Segoe UI", 10))
        subtitle.setAlignment(Qt.AlignmentFlag.AlignCenter)
        menu_layout.addWidget(subtitle)

        # Logo
        if os.path.exists(LOGO_PATH):
            logo_lbl = QLabel()
            pix = QPixmap(LOGO_PATH)
            pix = pix.scaledToWidth(140, Qt.TransformationMode.SmoothTransformation)
            logo_lbl.setPixmap(pix)
            logo_lbl.setAlignment(Qt.AlignmentFlag.AlignCenter)
            menu_layout.addWidget(logo_lbl)
        else:
            menu_layout.addSpacing(12)

        # Boutons menu
        self.menu_buttons = {}
        buttons = [
            ("students", "🎓 BD Étudiants"),
            ("families", "🏠 BD Familles"),
            ("nutrition", "🥗 BD Nutrition"),
            ("account", "👤 Mon Compte"),
            ("qr_scan", "📷 Scanner QR"),
            ("new_id", "➕ Nouvelle Identification"),
            ("print", "🖨 Imprimer Fiche"),
            ("import_excel", "📥 Importer Excel"),
        ]
        for key, label in buttons:
            btn = QPushButton(label)
            btn.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Fixed)
            self.menu_buttons[key] = btn
            menu_layout.addWidget(btn)

        self.menu_buttons["import_excel"].clicked.connect(self.import_excel_students)

        menu_layout.addStretch()

        # Bouton déconnexion (rouge) - placé en bas du menu
        self.btn_logout = QPushButton("🔒 Déconnexion")
        self.btn_logout.setStyleSheet("background-color:#c0392b;color:white;font-weight:bold;")
        self.btn_logout.clicked.connect(self.logout)
        menu_layout.addWidget(self.btn_logout)

        # Caméra compacte (sous le menu)
        cam_box = QVBoxLayout()
        cam_title = QLabel("Flux Caméra (300×200)")
        cam_title.setAlignment(Qt.AlignmentFlag.AlignCenter)
        cam_box.addWidget(cam_title)

        self.camera_frame = QFrame()
        self.camera_frame.setObjectName("cameraFrame")
        self.camera_frame.setFixedSize(300, 200)
        self.camera_frame_layout = QVBoxLayout(self.camera_frame)
        self.camera_frame_layout.setContentsMargins(0, 0, 0, 0)

        self.camera_label = QLabel()
        self.camera_label.setFixedSize(300, 200)
        self.camera_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.camera_frame_layout.addWidget(self.camera_label)
        cam_box.addWidget(self.camera_frame)

        cam_btns = QHBoxLayout()
        self.btn_start_camera = QPushButton("Démarrer Caméra")
        self.btn_stop_camera = QPushButton("Fermer Caméra")
        self.btn_stop_camera.setEnabled(False)
        cam_btns.addWidget(self.btn_start_camera)
        cam_btns.addWidget(self.btn_stop_camera)
        cam_box.addLayout(cam_btns)

        menu_layout.addLayout(cam_box)
        main_layout.addLayout(menu_layout, 1)

        # Contenu principal
        self.content_layout = QVBoxLayout()
        main_layout.addLayout(self.content_layout, 4)

        # Pré-création objets réutilisables (mais non ajoutés à content)
        self.qr_input = None
        self.btn_qr_pos = None
        self.qr_table = None
        self.btn_reset_qr = None

        # Connexions boutons menu
        self.menu_buttons["students"].clicked.connect(lambda: self.show_tab("students"))
        self.menu_buttons["families"].clicked.connect(lambda: self.show_tab("families"))
        self.menu_buttons["nutrition"].clicked.connect(lambda: self.show_tab("nutrition"))
        self.menu_buttons["account"].clicked.connect(lambda: self.show_tab("account"))
        self.menu_buttons["qr_scan"].clicked.connect(lambda: self.show_tab("qr_scan"))
        self.menu_buttons["new_id"].clicked.connect(self.open_new_id)
        self.menu_buttons["print"].clicked.connect(self.print_students)

        # Connexions caméra
        self.btn_start_camera.clicked.connect(self.start_camera)
        self.btn_stop_camera.clicked.connect(self.stop_camera)

        # Onglet par défaut
        self.show_tab("students")

    # ---------------------------
    # Logout
    # ---------------------------
    def logout(self):
        from gui.login_page import LoginPage
        # Ferme le panel actuel
        self.close()
        # Ouvre une nouvelle fenêtre login
        self.login_window = LoginPage()
        self.login_window.show()

    # ---------------------------
    # Onglets
    # ---------------------------
    def show_tab(self, tab_name):
        for k, btn in self.menu_buttons.items():
            btn.setStyleSheet("background-color:#0055aa;" if k == tab_name else "background-color:#2d2d2d;")
        self.clear_content()
        if tab_name == "students":
            self.show_students_tab()
        elif tab_name == "families":
            self.show_families_tab()
        elif tab_name == "nutrition":
            self.show_nutrition_tab()
        elif tab_name == "account":
            self.show_account_tab()
        elif tab_name == "qr_scan":
            self.show_qr_scan_tab()

    def clear_content(self):
        for i in reversed(range(self.content_layout.count())):
            item = self.content_layout.itemAt(i)
            if item:
                w = item.widget()
                if w:
                    w.setParent(None)

    def import_excel_students(self):
        # Connexion à la base pour récupérer les utilisateurs
        conn = sqlite3.connect(USERS_DB)
        c = conn.cursor()
        users = [row[0] for row in c.execute("SELECT username FROM users").fetchall()]
        conn.close()

        if not users:
            QMessageBox.warning(self, "Erreur", "Pas d'utilisateur trouvé. Créez un utilisateur d'abord.")
            return

       # Associer les données directement à l'utilisateur connecté
        owner = self.nom_utilisateur_actif

        # Sélection du fichier Excel
        file_path, _ = QFileDialog.getOpenFileName(
            self, "Importer fichier Excel", BASE_DIR, "Excel Files (*.xlsx *.xls)"
        )
        if not file_path:
            return

        try:
            df = pd.read_excel(file_path, dtype=str).fillna("")
        except Exception as e:
            QMessageBox.warning(self, "Erreur", f"Impossible de lire le fichier Excel : {e}")
            return

        # Progress dialog
        total = len(df)
        progress = QProgressDialog("Importation en cours...", "Annuler", 0, max(1, total), self)
        progress.setWindowTitle("Importation Excel")
        progress.setWindowModality(Qt.WindowModality.WindowModal)
        progress.setMinimumDuration(0)
        progress.show()

        # Mapping colonnes Excel -> DB
        mapping = {
            "Matricule": "matricule", "Nom": "nom", "Post-Nom": "postnom", "Prénom": "prenom",
            "Nom du Père": "pere", "Nom de la Mère": "mere", "Naissance": "naissance", "Sexe": "sexe",
            "Institution": "institution", "Classe": "classe", "Resultat": "resultat", "Mention": "mention",
            "PHOTO": "photo"
        }

        conn = sqlite3.connect(STUDENTS_DB)
        c = conn.cursor()
        inserted = 0

        for i, row in df.iterrows():
            if progress.wasCanceled():
                break
            progress.setValue(i)

            rec = {}
            for excel_col, db_col in mapping.items():
                found = next((orig for orig in df.columns if orig.strip().lower() == excel_col.strip().lower()), None)
                val = str(row[found]).strip() if found else ""

                if db_col == "photo":
                    photo_name = os.path.basename(val)
                    full_path = os.path.join(PHOTOS_DIR, photo_name)
                    if photo_name and os.path.exists(full_path):
                        crop_and_resize_photo(full_path, (300, 400))
                        rec["photo"] = photo_name
                    else:
                        rec["photo"] = ""
                else:
                    rec[db_col] = val

            # Association avec l'utilisateur qui a importé
            rec["owner"] = owner
            rec.setdefault("date_enreg", datetime.now().strftime("%Y-%m-%d %H:%M:%S"))
            rec.setdefault("statut", "actif")

            # Insertion dans la DB
            c.execute(f"PRAGMA table_info(students)")
            existing_cols = [r[1] for r in c.fetchall()]
            insert_cols = [col for col in existing_cols if col in rec]
            if not insert_cols:
                continue
            placeholders = ",".join(["?"] * len(insert_cols))
            cols_sql = ",".join(insert_cols)
            values = [rec[col] for col in insert_cols]

            try:
                c.execute(f"INSERT OR REPLACE INTO students ({cols_sql}) VALUES ({placeholders})", tuple(values))
            except Exception as e:
                print(f"Erreur insertion ligne {i}: {e}")

            inserted += 1

            # Génération ID / PDF
            data_for_card = {k: rec.get(k, "") for k in ["matricule","nom","postnom","prenom","pere","mere","naissance","sexe","institution","classe"]}
            data_for_card["photo"] = os.path.join(PHOTOS_DIR, rec.get("photo","")) if rec.get("photo") else ""
            try:
                generate_id_card(data_for_card)
                generate_pdf(data_for_card)
            except Exception as e:
                print("Erreur génération ID/PDF:", e)

        conn.commit()
        conn.close()
        progress.setValue(total)
        progress.close()

        # Recharge l'affichage des étudiants pour ce compte
        self.load_students()
        QMessageBox.information(self, "Terminé", f"Import terminé. Lignes traitées: {total} | insérées: {inserted}")


    # ---------------------------
    # BD étudiants
    # ---------------------------
    def show_students_tab(self):
        header = QLabel("BD Étudiants")
        header.setFont(QFont("Segoe UI", 18, QFont.Weight.Bold))
        header.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.content_layout.addWidget(header)

        self.students_table = QTableWidget()
        self.students_table.setColumnCount(12)
        self.students_table.setHorizontalHeaderLabels([
            "Matricule", "Nom", "Postnom", "Prénom", "Père", "Mère",
            "Naissance", "Sexe", "Institution", "Classe", "Résultat", "Mention"
        ])
        self.students_table.horizontalHeader().setStretchLastSection(True)
        self.content_layout.addWidget(self.students_table)
        self.load_students()

    def load_students(self):
        self.students_table.setRowCount(0)
        conn = sqlite3.connect(STUDENTS_DB)
        c = conn.cursor()
        try:
            if self.user_role == "admin":
                c.execute("""
                    SELECT matricule, nom, postnom, prenom, pere, mere, naissance, sexe,
                           institution, classe, resultat, mention
                    FROM students ORDER BY rowid DESC
                """)
                rows = c.fetchall()
            else:
                c.execute("""
                    SELECT matricule, nom, postnom, prenom, pere, mere, naissance, sexe,
                           institution, classe, resultat, mention
                    FROM students WHERE owner=? ORDER BY rowid DESC
                """, (self.nom_utilisateur_actif,))
                rows = c.fetchall()

            for i, row in enumerate(rows):
                self.students_table.insertRow(i)
                for j, val in enumerate(row):
                    item = QTableWidgetItem(str(val))
                    item.setFlags(Qt.ItemFlag.ItemIsEnabled | Qt.ItemFlag.ItemIsSelectable)
                    self.students_table.setItem(i, j, item)
        except Exception as e:
            QMessageBox.warning(self, "Erreur DB", f"Impossible de charger students: {e}")
        finally:
            conn.close()

    # ---------------------------
    # Families & Nutrition Tabs (minimal viewer)
    # ---------------------------
    def show_families_tab(self):
        header = QLabel("BD Familles")
        header.setFont(QFont("Segoe UI", 18, QFont.Weight.Bold))
        header.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.content_layout.addWidget(header)

        tbl = QTableWidget()
        tbl.setColumnCount(6)
        tbl.setHorizontalHeaderLabels(["Matricule", "Nom Famille", "Responsable", "Zone", "Nb Enfants", "Date Début"])
        tbl.horizontalHeader().setStretchLastSection(True)
        self.content_layout.addWidget(tbl)

        conn = sqlite3.connect(STUDENTS_DB)
        c = conn.cursor()
        try:
            c.execute("SELECT matricule, nom_famille, responsable, zone, nb_enfants, date_debut_suivi FROM families ORDER BY rowid DESC")
            rows = c.fetchall()
            for i, r in enumerate(rows):
                tbl.insertRow(i)
                for j, v in enumerate(r):
                    tbl.setItem(i, j, QTableWidgetItem(str(v)))
        except Exception as e:
            QMessageBox.warning(self, "Erreur DB", f"Impossible de charger families: {e}")
        finally:
            conn.close()

    def show_nutrition_tab(self):
        header = QLabel("BD Nutrition")
        header.setFont(QFont("Segoe UI", 18, QFont.Weight.Bold))
        header.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.content_layout.addWidget(header)

        tbl = QTableWidget()
        tbl.setColumnCount(6)
        tbl.setHorizontalHeaderLabels(["Num dossier", "Nom", "Sexe", "Date Naiss.", "Poids", "ScoreZ"])
        tbl.horizontalHeader().setStretchLastSection(True)
        self.content_layout.addWidget(tbl)

        conn = sqlite3.connect(STUDENTS_DB)
        c = conn.cursor()
        try:
            c.execute("SELECT num_dossier, nom, sexe, date_naissance, poids, scorez FROM nutrition ORDER BY rowid DESC")
            rows = c.fetchall()
            for i, r in enumerate(rows):
                tbl.insertRow(i)
                for j, v in enumerate(r):
                    tbl.setItem(i, j, QTableWidgetItem(str(v)))
        except Exception as e:
            QMessageBox.warning(self, "Erreur DB", f"Impossible de charger nutrition: {e}")
        finally:
            conn.close()

    # ---------------------------
    # Nouvelle identification
    # ---------------------------
    def open_new_id(self):
        try:
            self.new_id_window = StudentsPage(user_role=self.user_role)
            try:
                setattr(self.new_id_window, "nom_utilisateur_actif", self.nom_utilisateur_actif)
            except Exception:
                pass
            self.new_id_window.show()
        except Exception as e:
            QMessageBox.warning(self, "Erreur", f"Impossible d'ouvrir la page d'identification: {e}")

    # ---------------------------
    # Onglet QR scan (caméra + POS/USB)
    # ---------------------------
    def show_qr_scan_tab(self):
        self.clear_content()

        # Header
        header = QLabel("Scanner QR (Caméra intégrée ou POS)")
        header.setFont(QFont("Segoe UI", 16, QFont.Weight.Bold))
        header.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.content_layout.addWidget(header)

        # compteur cartes
        self.qr_count_label.setFont(QFont("Segoe UI", 14, QFont.Weight.Bold))
        self.qr_count_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.content_layout.addWidget(self.qr_count_label)

        # Champ de scan POS
        pos_layout = QHBoxLayout()
        if self.qr_input is None:
            self.qr_input = QLineEdit()
            self.qr_input.setPlaceholderText("Scanner QR ici ou coller le texte")
            self.qr_input.textChanged.connect(self._pos_input_changed)
        self.qr_input.setFocus()
        pos_layout.addWidget(self.qr_input)

        if self.btn_qr_pos is None:
            self.btn_qr_pos = QPushButton("Scanner POS")
            self.btn_qr_pos.clicked.connect(self._btn_pos_trigger)
        pos_layout.addWidget(self.btn_qr_pos)

        self.content_layout.addLayout(pos_layout)

        # Table QR
        if self.qr_table is None:
            self.qr_table = QTableWidget()
            self.qr_table.setColumnCount(5)
            self.qr_table.setHorizontalHeaderLabels(["Matricule", "Nom", "Post-nom", "Prénom", "Date/Heure"])
            self.qr_table.horizontalHeader().setStretchLastSection(True)
        self.content_layout.addWidget(self.qr_table)

        # Bouton reset QR
        if self.btn_reset_qr is None:
            self.btn_reset_qr = QPushButton("Réinitialiser Compteur (admin)")
            self.btn_reset_qr.setStyleSheet("background-color:#27ae60;color:white;font-weight:bold;")
            self.btn_reset_qr.clicked.connect(self.reset_qr_count)
        self.content_layout.addWidget(self.btn_reset_qr)

        # Recharge les données dans la table
        self.load_qr_table()

    # ---------------------------
    # Méthode pour scan automatique POS/USB
    # ---------------------------
    def _pos_input_changed(self):
        qr_data_raw = self.qr_input.text()
        if not qr_data_raw:
            return

        try:
            norm = normalize_scanned(qr_data_raw)
            if "|" in norm and norm.split("|", 1)[0].strip():
                self.process_qr_data(norm)
                self.qr_input.clear()
                self.qr_input.setFocus()
        except Exception as e:
            print(f"Erreur traitement QR POS (auto): {e}")

    def _btn_pos_trigger(self):
        qr_data_raw = self.qr_input.text()
        if not qr_data_raw:
            QApplication.beep()
            QMessageBox.warning(self, "Erreur", "Aucun QR détecté")
            return
        norm = normalize_scanned(qr_data_raw)
        self.process_qr_data(norm)
        self.qr_input.clear()
        self.qr_input.setFocus()

    # ---------------------------
    # POS / USB scan (bouton legacy)
    # ---------------------------
    def scan_qr_pos(self):
        if not self.qr_input:
            QApplication.beep()
            QMessageBox.warning(self, "Erreur", "Aucun champ de scan actif")
            return
        qr_data = self.qr_input.text().strip()
        if not qr_data:
            QApplication.beep()
            QMessageBox.warning(self, "Erreur", "Aucun QR détecté")
            return
        norm = normalize_scanned(qr_data)
        self.process_qr_data(norm)
        self.qr_input.clear()

    # ---------------------------
    # traitement QR
    # ---------------------------
    def process_qr_data(self, qr_data):
        try:
            if not qr_data:
                raise ValueError("Données QR vides")
            if "~" not in qr_data:
                qr_data = normalize_scanned(qr_data)

            parts = [p.strip() for p in qr_data.split("~", 1)]
            if len(parts) != 2:
                raise ValueError("Format QR invalide (attendu ID ~ Nom Post-nom Prénom)")

            matricule = parts[0]
            noms = parts[1].split()
            nom = noms[0] if len(noms) > 0 else ""
            postnom = noms[1] if len(noms) > 2 else (noms[1] if len(noms) == 2 else "")
            prenom = noms[2] if len(noms) > 2 else (noms[1] if len(noms) == 2 else "")
            timestamp = time.strftime("%Y-%m-%d %H:%M:%S")

            conn = sqlite3.connect(STUDENTS_DB)
            c = conn.cursor()
            c.execute("SELECT id FROM access_qr WHERE matricule=? LIMIT 1", (matricule,))
            if c.fetchone():
                QApplication.beep()
                QMessageBox.warning(self, "Carte déjà scannée", f"Matricule {matricule} a déjà été scanné.")
                conn.close()
                return

            c.execute(
                "INSERT INTO access_qr (matricule, nom, postnom, prenom, timestamp, owner) VALUES (?,?,?,?,?,?)",
                (matricule, nom, postnom, prenom, timestamp, self.nom_utilisateur_actif)
            )
            conn.commit()
            conn.close()

            self.load_qr_table()
            QApplication.beep()
            QMessageBox.information(self, "Accès OK", f"Accès autorisé pour {matricule}\n{nom} {postnom} {prenom}")
        except Exception as e:
            QApplication.beep()
            QMessageBox.warning(self, "Erreur QR", f"Impossible de traiter le QR: {e}")

    # ---------------------------
    # reset QR
    # ---------------------------
    def reset_qr_count(self):
        pwd, ok = QInputDialog.getText(self, "Mot de passe admin", "Entrez le mot de passe admin :", QLineEdit.EchoMode.Password)
        if not ok:
            return
        if pwd != self.admin_password:
            QApplication.beep()
            QMessageBox.warning(self, "Erreur", "Mot de passe incorrect")
            return
        conn = sqlite3.connect(STUDENTS_DB)
        c = conn.cursor()
        try:
            c.execute("DELETE FROM access_qr")
            conn.commit()
            if self.qr_table:
                self.qr_table.setRowCount(0)
            self.qr_count_label.setText("Cartes scannées : 0")
            QMessageBox.information(self, "Réinitialisation", "Compteur QR remis à zéro.")
        except Exception as e:
            QMessageBox.warning(self, "Erreur", f"Impossible de réinitialiser: {e}")
        finally:
            conn.close()

    # ---------------------------
    # Caméra handling
    # ---------------------------
    def start_camera(self):
        if self.camera_running:
            return
        try:
            self.cap = cv2.VideoCapture(0, cv2.CAP_DSHOW)
            if not self.cap.isOpened():
                self.cap.release()
                self.cap = cv2.VideoCapture(0)
            if not self.cap.isOpened():
                raise RuntimeError("Impossible d'ouvrir la caméra.")
            self.camera_running = True
            self.btn_start_camera.setEnabled(False)
            self.btn_stop_camera.setEnabled(True)
            self.camera_timer.start(30)
        except Exception as e:
            QMessageBox.warning(self, "Caméra", f"Erreur ouverture caméra: {e}")
            self.camera_running = False
            self.cap = None

    def stop_camera(self):
        if not self.camera_running:
            return
        self.camera_timer.stop()
        try:
            if self.cap:
                self.cap.release()
            self.cap = None
        except Exception:
            pass
        self.camera_running = False
        self.btn_start_camera.setEnabled(True)
        self.btn_stop_camera.setEnabled(False)
        self.camera_label.clear()
        self.camera_label.setText("Caméra arrêtée")

    def _camera_grab(self):
        if not self.cap:
            return
        ret, frame = self.cap.read()
        if not ret:
            return
        display = cv2.resize(frame, (300, 200))
        rgb = cv2.cvtColor(display, cv2.COLOR_BGR2RGB)
        h, w, ch = rgb.shape
        bytes_per_line = ch * w
        img = QImage(rgb.data, w, h, bytes_per_line, QImage.Format.Format_RGB888)
        pix = QPixmap.fromImage(img)
        self.camera_label.setPixmap(pix)

        try:
            qrs = decode(frame)
            if qrs:
                qr_text = qrs[0].data.decode("utf-8")
                norm = normalize_scanned(qr_text)
                self.process_qr_data(norm)
                QApplication.beep()
                self.stop_camera()
        except Exception:
            pass

    # ---------------------------
    # Imprimer étudiants (CSV) avec tri par date ou matricule
    # ---------------------------
    def print_students(self):
        try:
            conn = sqlite3.connect(STUDENTS_DB)
            df = pd.read_sql_query("SELECT * FROM students", conn)
            conn.close()
            if df.empty:
                QMessageBox.information(self, "Info", "Aucun étudiant à imprimer.")
                return

            # tri / filtre
            filter_options = ["Aucun", "Par Matricule", "Par Date"]
            choice, ok = QInputDialog.getItem(self, "Filtrer / Trier", "Choisissez un filtre:", filter_options, 0, False)
            if ok and choice:
                if choice == "Par Matricule":
                    mat, ok2 = QInputDialog.getText(self, "Matricule", "Saisir matricule:")
                    if ok2 and mat:
                        df = df[df["matricule"] == mat]
                elif choice == "Par Date":
                    start, ok1 = QInputDialog.getText(self, "Date début", "Saisir date début (YYYY-MM-DD):")
                    end, ok2 = QInputDialog.getText(self, "Date fin", "Saisir date fin (YYYY-MM-DD):")
                    if ok1 and ok2:
                        df["naissance"] = pd.to_datetime(df["naissance"], errors='coerce')
                        df = df[(df["naissance"] >= start) & (df["naissance"] <= end)]

            # export CSV
            export_path = QFileDialog.getSaveFileName(self, "Enregistrer CSV", BASE_DIR, "CSV Files (*.csv)")[0]
            if export_path:
                df.to_csv(export_path, index=False)
                QMessageBox.information(self, "Export CSV", f"Fichier exporté : {export_path}")
        except Exception as e:
            QMessageBox.warning(self, "Erreur impression", f"Impossible d'imprimer/exporter: {e}")

    # ---------------------------
    # Mon Compte : changement & statistiques
    # ---------------------------
    def show_account_tab(self):
        self.clear_content()

        # Header
        if not hasattr(self, "account_header"):
            self.account_header = QLabel(f"Mon Compte - {self.nom_utilisateur_actif}")
            self.account_header.setFont(QFont("Segoe UI", 16, QFont.Weight.Bold))
            self.account_header.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.content_layout.addWidget(self.account_header)

        # Photo profil + upload
        if not hasattr(self, "profile_pic_label"):
            self.profile_pic_label = QLabel()
            self.profile_pic_label.setFixedSize(120, 140)
            self.profile_pic_label.setStyleSheet("background:#222;border-radius:6px;")
            self.upload_btn = QPushButton("⬆️ Upload Photo Profil")
            self.upload_btn.clicked.connect(self.upload_profile_photo)
            self.photo_box = QHBoxLayout()
            self.photo_box.addWidget(self.profile_pic_label)
            self.photo_box.addWidget(self.upload_btn)
        self.content_layout.addLayout(self.photo_box)

        # Changement identifiants
        if not hasattr(self, "username_input"):
            self.username_input = QLineEdit()
            self.username_input.setPlaceholderText("Nouveau nom d’utilisateur")
            self.password_input = QLineEdit()
            self.password_input.setPlaceholderText("Nouveau mot de passe")
            self.password_input.setEchoMode(QLineEdit.EchoMode.Password)
            self.btn_save_account = QPushButton("💾 Enregistrer les modifications")
            self.btn_save_account.clicked.connect(self.save_account_changes)
            self.form_box = QVBoxLayout()
            self.form_box.addWidget(QLabel("Changer le nom d’utilisateur :"))
            self.form_box.addWidget(self.username_input)
            self.form_box.addWidget(QLabel("Changer le mot de passe :"))
            self.form_box.addWidget(self.password_input)
            self.form_box.addWidget(self.btn_save_account)
            self.form_frame = QFrame()
            self.form_frame.setLayout(self.form_box)
            self.form_frame.setStyleSheet("QFrame {background:#2d2d2d;border-radius:8px;padding:10px;}")
        self.content_layout.addWidget(self.form_frame)

        # Statistiques
        if not hasattr(self, "btn_stat_students"):
            self.stats_header = QLabel("📊 Statistiques de votre compte")
            self.stats_header.setFont(QFont("Segoe UI", 14, QFont.Weight.Bold))
            self.stats_header.setAlignment(Qt.AlignmentFlag.AlignCenter)
            self.content_layout.addWidget(self.stats_header)

            self.btn_stat_students = QPushButton("Étudiants\n0")
            self.btn_stat_families = QPushButton("Familles\n0")
            self.btn_stat_nutrition = QPushButton("Nutrition\n0")
            self.btn_stat_qr = QPushButton("Cartes scannées\n0")
            for b in (self.btn_stat_students, self.btn_stat_families, self.btn_stat_nutrition, self.btn_stat_qr):
                b.setFixedHeight(90)
                b.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Fixed)
                b.setStyleSheet("font-size:14px;font-weight:bold;")
            self.stats_layout = QHBoxLayout()
            self.stats_layout.addWidget(self.btn_stat_students)
            self.stats_layout.addWidget(self.btn_stat_families)
            self.stats_layout.addWidget(self.btn_stat_nutrition)
            self.stats_layout.addWidget(self.btn_stat_qr)

            # connect once
            self.btn_stat_students.clicked.connect(lambda: self.show_tab("students"))
            self.btn_stat_families.clicked.connect(lambda: self.show_tab("families"))
            self.btn_stat_nutrition.clicked.connect(lambda: self.show_tab("nutrition"))
            self.btn_stat_qr.clicked.connect(lambda: self.show_tab("qr_scan"))

        self.content_layout.addLayout(self.stats_layout)

        # load stats et photo
        self.load_user_stats()
        self.load_profile_photo()

    # ---------------------------
    # Recharge la table QR
    # ---------------------------
    def load_qr_table(self):
        if not hasattr(self, "qr_table") or self.qr_table is None:
            return
        self.qr_table.setRowCount(0)
        conn = sqlite3.connect(STUDENTS_DB)
        c = conn.cursor()
        try:
            if self.user_role == "admin":
                c.execute("SELECT matricule, nom, postnom, prenom, timestamp FROM access_qr ORDER BY id ASC")
            else:
                c.execute(
                    "SELECT matricule, nom, postnom, prenom, timestamp FROM access_qr WHERE owner=? ORDER BY id ASC",
                    (self.nom_utilisateur_actif,)
                )
            rows = c.fetchall()
            for row_idx, row_data in enumerate(rows):
                self.qr_table.insertRow(row_idx)
                for col_idx, value in enumerate(row_data):
                    self.qr_table.setItem(row_idx, col_idx, QTableWidgetItem(str(value)))
            self.qr_count_label.setText(f"Cartes scannées : {len(rows)}")
            # update account stats button if visible
            try:
                self.btn_stat_qr.setText(f"Cartes scannées\n{len(rows)}")
            except Exception:
                pass
        except Exception as e:
            QMessageBox.warning(self, "Erreur DB", f"Impossible de charger access_qr: {e}")
        finally:
            conn.close()

    def upload_profile_photo(self):
        path, _ = QFileDialog.getOpenFileName(self, "Choisir une photo", BASE_DIR, "Images (*.png *.jpg *.jpeg)")
        if not path:
            return
        _, ext = os.path.splitext(path)
        dest = os.path.join(PROFILE_PHOTOS_DIR, f"{self.nom_utilisateur_actif}{ext}")
        try:
            shutil.copy(path, dest)
            crop_and_resize_photo(dest, (300, 400))  # si tu as cette fonction
            conn = sqlite3.connect(USERS_DB)
            c = conn.cursor()
            c.execute("CREATE TABLE IF NOT EXISTS users (username TEXT PRIMARY KEY, password TEXT, role TEXT DEFAULT 'user', profile_photo TEXT)")
            c.execute("UPDATE users SET profile_photo=? WHERE username=?", (dest, self.nom_utilisateur_actif))
            if not c.rowcount:
                c.execute("INSERT OR REPLACE INTO users (username, profile_photo) VALUES (?,?)", (self.nom_utilisateur_actif, dest))
            conn.commit()
            conn.close()
            self.profile_photo_path = dest
            self.load_profile_photo()
            QMessageBox.information(self, "Photo sauvegardée", "Photo de profil enregistrée.")
        except Exception as e:
            QMessageBox.warning(self, "Erreur photo", f"Impossible d'enregistrer la photo: {e}")

    def load_profile_photo(self):
        if not hasattr(self, "profile_pic_label"):
            return
        conn = sqlite3.connect(USERS_DB)
        c = conn.cursor()
        try:
            c.execute("CREATE TABLE IF NOT EXISTS users (username TEXT PRIMARY KEY, password TEXT, role TEXT DEFAULT 'user', profile_photo TEXT)")
            c.execute("SELECT profile_photo FROM users WHERE username=? LIMIT 1", (self.nom_utilisateur_actif,))
            row = c.fetchone()
            if row and row[0] and os.path.exists(row[0]):
                pix = QPixmap(row[0]).scaled(self.profile_pic_label.width(), self.profile_pic_label.height(), Qt.AspectRatioMode.KeepAspectRatio, Qt.TransformationMode.SmoothTransformation)
                self.profile_pic_label.setPixmap(pix)
                self.profile_photo_path = row[0]
            else:
                self.profile_pic_label.clear()
                self.profile_pic_label.setText("Aucune photo")
        except Exception:
            pass
        finally:
            conn.close()


    def save_account_changes(self):
        new_username = self.username_input.text().strip()
        new_password = self.password_input.text().strip()

        if not new_username and not new_password:
            QMessageBox.information(self, "Info", "Aucun changement détecté.")
            return

        conn = sqlite3.connect(USERS_DB)
        c = conn.cursor()
        try:
            c.execute("""
                CREATE TABLE IF NOT EXISTS users (
                    username TEXT PRIMARY KEY,
                    password TEXT,
                    role TEXT DEFAULT 'user',
                    profile_photo TEXT
                )
            """)

            c.execute("SELECT username FROM users WHERE username=?", (self.nom_utilisateur_actif,))
            existing = c.fetchone()
            if existing:
                if new_username:
                    c.execute("SELECT username FROM users WHERE username=?", (new_username,))
                    if c.fetchone():
                        QMessageBox.warning(self, "Erreur", "Le nouveau nom d'utilisateur existe déjà.")
                        conn.close()
                        return
                    c.execute("UPDATE users SET username=? WHERE username=?", (new_username, self.nom_utilisateur_actif))
                    # Mettre à jour owner dans students et access_qr
                    conn_students = sqlite3.connect(STUDENTS_DB)
                    c2 = conn_students.cursor()
                    c2.execute("UPDATE students SET owner=? WHERE owner=?", (new_username, self.nom_utilisateur_actif))
                    c2.execute("UPDATE access_qr SET owner=? WHERE owner=?", (new_username, self.nom_utilisateur_actif))
                    c2.execute("UPDATE families SET owner=? WHERE owner=?", (new_username, self.nom_utilisateur_actif))
                    conn_students.commit()
                    conn_students.close()
                    self.nom_utilisateur_actif = new_username
                    self.title_label.setText(f"CONNECTÉ : {self.nom_utilisateur_actif}")
                if new_password:
                    c.execute("UPDATE users SET password=? WHERE username=?", (new_password, self.nom_utilisateur_actif))
            else:
                username_to_create = new_username or self.nom_utilisateur_actif
                password_to_create = new_password or ""
                c.execute("INSERT OR REPLACE INTO users (username, password) VALUES (?, ?)",
                          (username_to_create, password_to_create))
                if new_username:
                    self.nom_utilisateur_actif = username_to_create
                    self.title_label.setText(f"CONNECTÉ : {self.nom_utilisateur_actif}")

            conn.commit()
            QMessageBox.information(self, "Succès", "Informations mises à jour.")
        except Exception as e:
            QMessageBox.warning(self, "Erreur", f"Impossible de modifier le compte : {e}")
        finally:
            conn.close()
            self.load_user_stats()
            self.load_profile_photo()

    def load_user_stats(self):
        conn = sqlite3.connect(STUDENTS_DB)
        c = conn.cursor()
        try:
            user_filter = ""
            params = ()
            if self.user_role != "admin":
                user_filter = "WHERE owner=?"
                params = (self.nom_utilisateur_actif,)

            c.execute(f"SELECT COUNT(*) FROM students {user_filter}", params)
            nb_students = c.fetchone()[0]
            c.execute(f"SELECT COUNT(*) FROM access_qr {user_filter}", params)
            nb_qr = c.fetchone()[0]
            c.execute(f"SELECT COUNT(*) FROM families {user_filter}", params)
            nb_familles = c.fetchone()[0]
            c.execute(f"SELECT COUNT(*) FROM nutrition", ())
            nb_nutrition = c.fetchone()[0]

            # update big stat buttons if present
            try:
                self.btn_stat_students.setText(f"Étudiants\n{nb_students}")
                self.btn_stat_families.setText(f"Familles\n{nb_familles}")
                self.btn_stat_nutrition.setText(f"Nutrition\n{nb_nutrition}")
                self.btn_stat_qr.setText(f"Cartes scannées\n{nb_qr}")
            except Exception:
                pass

            # update small stats label if exists
            try:
                self.stats_label.setText(
                    f"👩‍🎓 Étudiants identifiés : <b>{nb_students}</b><br>"
                    f"🏠 Familles : <b>{nb_familles}</b><br>"
                    f"🥗 Nutrition : <b>{nb_nutrition}</b><br>"
                    f"💳 Cartes scannées : <b>{nb_qr}</b>"
                )
            except Exception:
                pass
        except Exception as e:
            QMessageBox.warning(self, "Erreur", f"Impossible de charger les statistiques : {e}")
        finally:
            conn.close()

# ---------------------------
# Exécution directe
# ---------------------------
if __name__ == "__main__":
    app = QApplication(sys.argv)
    w = StudentPanel(user_role="user", nom_utilisateur_actif="UtilisateurTest", admin_password="admin123")
    w.show()
    sys.exit(app.exec())
