# gui/qr_scan_page.py
import os
import cv2
import sqlite3
from PyQt6.QtWidgets import (
    QWidget, QLabel, QPushButton, QVBoxLayout, QHBoxLayout, QMessageBox, QListWidget, QListWidgetItem
)
from PyQt6.QtGui import QPixmap, QImage
from PyQt6.QtCore import Qt, QTimer
import qrcode

DB_PATH = os.path.join(os.getcwd(), "database", "students.db")

class QRScanPage(QWidget):
    """
    Page pour scanner QR code via webcam ou POS et afficher les informations pour autorisation.
    """

    def __init__(self, parent=None):
        super().__init__(parent)
        self.cap = None
        self.timer = QTimer(self)
        self.timer.timeout.connect(self._query_frame)
        self.current_frame = None
        self.last_qr_data = None

        main_layout = QVBoxLayout(self)
        main_layout.setContentsMargins(8, 8, 8, 8)
        main_layout.setSpacing(12)

        # Zone d'affichage vidéo
        self.video_label = QLabel("Caméra inactive")
        self.video_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.video_label.setFixedSize(400, 300)
        self.video_label.setStyleSheet("QLabel { border: 2px dashed #999; background: #fafafa; }")
        main_layout.addWidget(self.video_label, alignment=Qt.AlignmentFlag.AlignHCenter)

        # Boutons caméra
        btn_layout = QHBoxLayout()
        self.btn_open_camera = QPushButton("Ouvrir Caméra")
        self.btn_close_camera = QPushButton("Fermer Caméra")
        self.btn_scan_qr = QPushButton("Scanner QR")
        self.btn_scan_qr.setEnabled(False)
        btn_layout.addWidget(self.btn_open_camera)
        btn_layout.addWidget(self.btn_scan_qr)
        btn_layout.addWidget(self.btn_close_camera)
        main_layout.addLayout(btn_layout)

        # Zone info QR
        self.info_label = QLabel("")
        self.info_label.setWordWrap(True)
        main_layout.addWidget(self.info_label)

        # Liste des élèves scannés
        self.scanned_list = QListWidget()
        main_layout.addWidget(QLabel("Élèves autorisés :"))
        main_layout.addWidget(self.scanned_list)

        # Connexions
        self.btn_open_camera.clicked.connect(self.start_camera)
        self.btn_close_camera.clicked.connect(self.stop_camera)
        self.btn_scan_qr.clicked.connect(self.scan_qr)

    def start_camera(self):
        if self.cap is not None:
            return
        self.cap = cv2.VideoCapture(0, cv2.CAP_DSHOW)
        if not self.cap.isOpened():
            self.cap.release()
            self.cap = cv2.VideoCapture(0)
        if not self.cap.isOpened():
            QMessageBox.warning(self, "Caméra", "Impossible d'ouvrir la caméra.")
            self.cap = None
            return
        self.timer.start(30)
        self.btn_open_camera.setEnabled(False)
        self.btn_scan_qr.setEnabled(True)
        self.btn_close_camera.setEnabled(True)
        self.info_label.setText("Caméra ouverte. Cliquez sur Scanner QR.")

    def _query_frame(self):
        if not self.cap:
            return
        ret, frame = self.cap.read()
        if not ret:
            return
        self.current_frame = frame
        try:
            frame_rgb = cv2.cvtColor(frame, cv2.COLOR_BGR2RGB)
            h, w, ch = frame_rgb.shape
            bytes_per_line = ch * w
            qimg = QImage(frame_rgb.data, w, h, bytes_per_line, QImage.Format.Format_RGB888)
            pix = QPixmap.fromImage(qimg).scaled(
                self.video_label.width(),
                self.video_label.height(),
                Qt.AspectRatioMode.KeepAspectRatioByExpanding,
                Qt.TransformationMode.SmoothTransformation
            )
            self.video_label.setPixmap(pix)
        except Exception:
            pass

    def scan_qr(self):
        if self.current_frame is None:
            QMessageBox.information(self, "QR Scan", "Aucune image disponible.")
            return

        detector = cv2.QRCodeDetector()
        data, bbox, _ = detector.detectAndDecode(self.current_frame)
        if data:
            self.last_qr_data = data
            self.info_label.setText(f"QR détecté : {data}")
            # Chercher l'étudiant correspondant dans la DB
            conn = sqlite3.connect(DB_PATH)
            c = conn.cursor()
            c.execute("SELECT matricule, nom, postnom, prenom, classe FROM students WHERE matricule=?", (data,))
            row = c.fetchone()
            conn.close()
            if row:
                item_text = f"{row[0]} | {row[1]} {row[2]} {row[3]} | {row[4]}"
                item = QListWidgetItem(item_text)
                self.scanned_list.addItem(item)
                QMessageBox.information(self, "Autorisation", f"Élève autorisé : {row[1]} {row[2]} {row[3]}")
            else:
                QMessageBox.warning(self, "QR Inconnu", "Aucun élève trouvé pour ce QR.")
        else:
            QMessageBox.information(self, "QR Scan", "Aucun QR code détecté sur l'image.")

    def stop_camera(self):
        if self.timer.isActive():
            self.timer.stop()
        if self.cap:
            try:
                self.cap.release()
            except Exception:
                pass
            self.cap = None
        self.btn_open_camera.setEnabled(True)
        self.btn_scan_qr.setEnabled(False)
        self.btn_close_camera.setEnabled(False)
        self.info_label.setText("Caméra fermée.")
