# login_page.py
import sys
import os
import sqlite3
import smtplib
from email.message import EmailMessage
from PyQt6.QtWidgets import (
    QWidget, QLabel, QLineEdit, QPushButton,
    QVBoxLayout, QComboBox, QMessageBox, QSpacerItem, QSizePolicy,
    QDialog
)
from PyQt6.QtGui import QPixmap, QCursor
from PyQt6.QtCore import Qt

from gui.admin_panel import AdminPanel
from gui.student_panel import StudentPanel

BASE_DIR = os.getcwd()
USERS_DB = os.path.join(BASE_DIR, "database", "users.db")
LOGO_PATH = os.path.join(BASE_DIR, "logo.png")

# --------------------------
# Dialog pour changer le mot de passe
# --------------------------
class ChangePasswordDialog(QDialog):
    def __init__(self, username, parent=None):
        super().__init__(parent)
        self.username = username
        self.setWindowTitle("Modifier le mot de passe")
        self.setFixedSize(420, 300)

        self.setStyleSheet("""
            QWidget {background-color:#001a52; color:#e0e1dd; font-family:Segoe UI;}
            QLineEdit {padding:10px; border-radius:6px; border:1px solid #415a77; background:white; color:#000;}
            QPushButton {background-color:#00a7de; padding:10px; border-radius:6px; color:white; font-weight:bold;}
            QPushButton:hover {background-color:#007298;}
        """)

        layout = QVBoxLayout(self)
        title = QLabel(f"Modifier le mot de passe ({self.username})")
        title.setAlignment(Qt.AlignmentFlag.AlignCenter)
        title.setStyleSheet("font-size:20px; font-weight:bold; color:#e0e1dd;")
        layout.addWidget(title)

        self.old_pwd = QLineEdit()
        self.old_pwd.setPlaceholderText("Ancien mot de passe")
        self.old_pwd.setEchoMode(QLineEdit.EchoMode.Password)
        layout.addWidget(self.old_pwd)

        self.new_pwd = QLineEdit()
        self.new_pwd.setPlaceholderText("Nouveau mot de passe")
        self.new_pwd.setEchoMode(QLineEdit.EchoMode.Password)
        layout.addWidget(self.new_pwd)

        self.confirm_pwd = QLineEdit()
        self.confirm_pwd.setPlaceholderText("Confirmer le nouveau mot de passe")
        self.confirm_pwd.setEchoMode(QLineEdit.EchoMode.Password)
        layout.addWidget(self.confirm_pwd)

        btn = QPushButton("Modifier")
        btn.clicked.connect(self.update_password)
        layout.addWidget(btn)

    def update_password(self):
        old = self.old_pwd.text().strip()
        new = self.new_pwd.text().strip()
        confirm = self.confirm_pwd.text().strip()

        if not old or not new or not confirm:
            QMessageBox.warning(self, "Erreur", "Tous les champs sont obligatoires.")
            return

        if new != confirm:
            QMessageBox.warning(self, "Erreur", "Les mots de passe ne correspondent pas.")
            return

        conn = sqlite3.connect(USERS_DB)
        c = conn.cursor()
        c.execute("SELECT password FROM users WHERE username=?", (self.username,))
        row = c.fetchone()

        if not row or row[0] != old:
            QMessageBox.critical(self, "Erreur", "Ancien mot de passe incorrect.")
            conn.close()
            return

        c.execute("UPDATE users SET password=? WHERE username=?", (new, self.username))
        conn.commit()
        conn.close()
        QMessageBox.information(self, "Succès", "Mot de passe modifié avec succès.")
        self.accept()

# --------------------------
# Dialog pour mot de passe oublié
# --------------------------
class ForgotPasswordDialog(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Mot de passe oublié")
        self.setFixedSize(450, 320)

        self.setStyleSheet("""
            QWidget {background-color:#001a52; color:#e0e1dd; font-family:Segoe UI;}
            QLineEdit {padding:10px; border-radius:6px; border:1px solid #415a77; background:white; color:#000;}
            QPushButton {background-color:#00a7de; padding:10px; border-radius:6px; color:white; font-weight:bold;}
            QPushButton:hover {background-color:#007298;}
        """)

        layout = QVBoxLayout(self)
        title = QLabel("Demande de réinitialisation")
        title.setAlignment(Qt.AlignmentFlag.AlignCenter)
        title.setStyleSheet("font-size:18px; font-weight:bold;")
        layout.addWidget(title)

        # Champs
        self.username_input = QLineEdit()
        self.username_input.setPlaceholderText("Nom d'utilisateur")
        layout.addWidget(self.username_input)

        self.email_input = QLineEdit()
        self.email_input.setPlaceholderText("Adresse e-mail")
        layout.addWidget(self.email_input)

        self.temp_pwd_input = QLineEdit()
        self.temp_pwd_input.setPlaceholderText("Mot de passe temporaire")
        self.temp_pwd_input.setEchoMode(QLineEdit.EchoMode.Password)
        layout.addWidget(self.temp_pwd_input)

        btn = QPushButton("Envoyer la demande")
        btn.clicked.connect(self.send_request)
        layout.addWidget(btn)

    def send_request(self):
        uname = self.username_input.text().strip()
        email_user = self.email_input.text().strip()
        temp_pwd = self.temp_pwd_input.text().strip()

        if not uname or not email_user or not temp_pwd:
            QMessageBox.critical(self, "Erreur", "Impossible de modifier le mot de passe – user and password not accepted")
            return

        # Vérification utilisateur dans la base
        conn = sqlite3.connect(USERS_DB)
        c = conn.cursor()
        c.execute("SELECT username FROM users WHERE username=?", (uname,))
        if not c.fetchone():
            QMessageBox.critical(self, "Erreur", "Impossible de modifier le mot de passe – user and password not accepted")
            conn.close()
            return
        conn.close()

        # Email du technicien
        TECH_EMAIL = "musagaradaniel@gmail.com"
        SMTP_SERVER = "smtp.gmail.com"
        SMTP_PORT = 587
        SMTP_USER = "luxacademia.sc@gmail.com"       # Remplacer par un vrai email expéditeur
        SMTP_PASS = "Musb.12Dany"            # Remplacer par mot de passe ou app password

        try:
            msg = EmailMessage()
            msg['Subject'] = f"Demande réinitialisation mot de passe: {uname}"
            msg['From'] = SMTP_USER
            msg['To'] = TECH_EMAIL
            msg.set_content(
                f"Utilisateur : {uname}\n"
                f"Email utilisateur : {email_user}\n"
                f"Mot de passe temporaire proposé : {temp_pwd}\n\n"
                "Merci de procéder à la réinitialisation manuelle."
            )

            server = smtplib.SMTP(SMTP_SERVER, SMTP_PORT)
            server.starttls()
            server.login(SMTP_USER, SMTP_PASS)
            server.send_message(msg)
            server.quit()

            QMessageBox.information(
                self,
                "Demande envoyée",
                f"La demande a été envoyée au technicien ({TECH_EMAIL})."
            )
            self.accept()
        except Exception as e:
            QMessageBox.critical(self, "Erreur", f"Impossible d'envoyer le mail.\n{str(e)}")

# --------------------------
# Page de connexion principale
# --------------------------
class LoginPage(QWidget):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Connexion - Redeeming Love Ministries")
        self.setFixedSize(480, 620)

        self.setStyleSheet("""
            QWidget {background-color: #001a52; color: #e0e1dd; font-family: Segoe UI;}
            QLineEdit, QComboBox {padding: 12px; border-radius: 8px; border: 1px solid #415a77; background: #fff; color:#000;}
            QPushButton {background-color:#00a7de; padding:12px; border-radius:8px; color:white; font-weight:bold;}
            QPushButton:hover {background-color:#007298;}
            QLabel#title {font-size:28px; font-weight:bold; color:#e0e1dd;}
            QLabel.clickable {color:#00d5ff; font-size:12px;}
            QLabel.clickable:hover {color:#7fe8ff; text-decoration:underline;}
            QLabel#footer {color:#778da9; font-size:10px;}
        """)

        layout = QVBoxLayout()
        layout.setContentsMargins(50, 20, 50, 20)
        layout.setSpacing(15)
        layout.setAlignment(Qt.AlignmentFlag.AlignTop)
        self.setLayout(layout)

        if os.path.exists(LOGO_PATH):
            logo_label = QLabel()
            pixmap = QPixmap(LOGO_PATH).scaled(180, 180, Qt.AspectRatioMode.KeepAspectRatio,
                                               Qt.TransformationMode.SmoothTransformation)
            logo_label.setPixmap(pixmap)
            logo_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
            layout.addWidget(logo_label)

        title = QLabel("Connexion")
        title.setObjectName("title")
        title.setAlignment(Qt.AlignmentFlag.AlignCenter)
        layout.addWidget(title)

        self.username_combo = QComboBox()
        self.username_combo.setEditable(False)
        self.load_users()
        layout.addWidget(self.username_combo)

        self.password_input = QLineEdit()
        self.password_input.setPlaceholderText("Mot de passe")
        self.password_input.setEchoMode(QLineEdit.EchoMode.Password)
        layout.addWidget(self.password_input)

        self.login_btn = QPushButton("Se connecter")
        self.login_btn.clicked.connect(self.check_login)
        layout.addWidget(self.login_btn)

        # --- Modifier mot de passe
        self.change_pwd_label = QLabel("<u>Modifier le mot de passe</u>")
        self.change_pwd_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.change_pwd_label.setCursor(QCursor(Qt.CursorShape.PointingHandCursor))
        self.change_pwd_label.setProperty("class", "clickable")
        self.change_pwd_label.mousePressEvent = self.open_change_password
        layout.addWidget(self.change_pwd_label)

        # --- Mot de passe oublié
        self.forgot_pwd_label = QLabel("<u>Mot de passe oublié ?</u>")
        self.forgot_pwd_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.forgot_pwd_label.setCursor(QCursor(Qt.CursorShape.PointingHandCursor))
        self.forgot_pwd_label.setProperty("class", "clickable")
        self.forgot_pwd_label.mousePressEvent = self.open_forgot_password
        layout.addWidget(self.forgot_pwd_label)

        self.feedback = QLabel("")
        self.feedback.setAlignment(Qt.AlignmentFlag.AlignCenter)
        layout.addWidget(self.feedback)

        layout.addItem(QSpacerItem(20, 40, QSizePolicy.Policy.Minimum, QSizePolicy.Policy.Expanding))

        footer = QLabel("Created for RLM by DTech / D-Square Technologie © 2025 | Musagara Daniel")
        footer.setObjectName("footer")
        footer.setAlignment(Qt.AlignmentFlag.AlignCenter)
        layout.addWidget(footer)

    # ------------------------
    def load_users(self):
        self.username_combo.clear()
        if not os.path.exists(USERS_DB):
            return
        conn = sqlite3.connect(USERS_DB)
        c = conn.cursor()
        try:
            c.execute("SELECT username FROM users")
            self.username_combo.addItems([x[0] for x in c.fetchall()])
        except:
            pass
        conn.close()

    def check_login(self):
        uname = self.username_combo.currentText().strip()
        pwd = self.password_input.text().strip()
        conn = sqlite3.connect(USERS_DB)
        c = conn.cursor()
        c.execute("SELECT role FROM users WHERE username=? AND password=?", (uname, pwd))
        row = c.fetchone()
        conn.close()

        if row:
            self.open_panel(row[0], uname)
        else:
            self.feedback.setText("Nom d'utilisateur ou mot de passe incorrect.")

    def open_panel(self, role, uname):
        self.close()
        if role == "admin":
            self.panel = AdminPanel()
        else:
            self.panel = StudentPanel(user_role=role, username=uname)
        self.panel.show()

    def open_change_password(self, event):
        username = self.username_combo.currentText().strip()
        if not username:
            QMessageBox.warning(self, "Erreur", "Veuillez sélectionner un utilisateur.")
            return
        dialog = ChangePasswordDialog(username, self)
        dialog.exec()

    def open_forgot_password(self, event):
        dialog = ForgotPasswordDialog(self)
        dialog.exec()

# ---------------------------------------------
#               LANCEMENT
# ---------------------------------------------
if __name__ == "__main__":
    from PyQt6.QtWidgets import QApplication
    app = QApplication(sys.argv)
    login = LoginPage()
    login.show()
    sys.exit(app.exec())
