# gui/families_page.py
import os
import cv2
from datetime import datetime
import sqlite3

from PyQt6.QtWidgets import (
    QWidget, QLabel, QLineEdit, QPushButton, QVBoxLayout, QHBoxLayout,
    QFormLayout, QMessageBox, QListWidget, QListWidgetItem, QSizePolicy
)
from PyQt6.QtGui import QPixmap, QImage
from PyQt6.QtCore import Qt, QTimer

DB_PATH = os.path.join(os.getcwd(), "database", "families.db")
os.makedirs(os.path.join(os.getcwd(), "database"), exist_ok=True)

# ---------------------------
# Initialisation DB si inexistante
# ---------------------------
def init_db():
    conn = sqlite3.connect(DB_PATH)
    c = conn.cursor()
    c.execute("""
        CREATE TABLE IF NOT EXISTS families (
            id TEXT PRIMARY KEY,
            nom_responsable TEXT,
            prenom_responsable TEXT,
            adresse TEXT,
            contact TEXT,
            photo TEXT
        )
    """)
    conn.commit()
    conn.close()

init_db()

# ---------------------------
# Classe FamiliesPage
# ---------------------------
class FamiliesPage(QWidget):
    def __init__(self, parent=None):
        super().__init__(parent)

        self.cap = None
        self.timer = QTimer(self)
        self.timer.timeout.connect(self._query_frame)
        self.current_frame = None
        self.last_photo_path = None

        main_layout = QHBoxLayout(self)
        main_layout.setContentsMargins(8,8,8,8)
        main_layout.setSpacing(12)

        # ---------------------------
        # Formulaire famille
        # ---------------------------
        form_layout = QFormLayout()
        form_layout.setLabelAlignment(Qt.AlignmentFlag.AlignRight)
        form_layout.setFormAlignment(Qt.AlignmentFlag.AlignLeft | Qt.AlignmentFlag.AlignTop)

        self.input_id = QLineEdit()
        self.input_nom = QLineEdit()
        self.input_prenom = QLineEdit()
        self.input_adresse = QLineEdit()
        self.input_contact = QLineEdit()

        form_layout.addRow("ID:", self.input_id)
        form_layout.addRow("Nom Responsable:", self.input_nom)
        form_layout.addRow("Prénom Responsable:", self.input_prenom)
        form_layout.addRow("Adresse:", self.input_adresse)
        form_layout.addRow("Contact:", self.input_contact)

        actions_layout = QHBoxLayout()
        self.btn_save = QPushButton("Enregistrer")
        self.btn_clear = QPushButton("Effacer")
        actions_layout.addWidget(self.btn_save)
        actions_layout.addWidget(self.btn_clear)
        form_layout.addRow(actions_layout)

        left_widget = QWidget()
        left_widget.setLayout(form_layout)
        left_widget.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Preferred)
        main_layout.addWidget(left_widget, stretch=2)

        # ---------------------------
        # Photo + camera
        # ---------------------------
        right_layout = QVBoxLayout()
        right_layout.setSpacing(8)

        self.photo_label = QLabel("Aucune photo")
        self.photo_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        self.photo_label.setFixedSize(260, 320)
        self.photo_label.setStyleSheet("QLabel { border: 2px dashed #999; background: #fafafa; }")
        right_layout.addWidget(self.photo_label, alignment=Qt.AlignmentFlag.AlignHCenter)

        camera_btns = QHBoxLayout()
        self.btn_open_camera = QPushButton("Ouvrir Caméra")
        self.btn_capture = QPushButton("Capturer Photo")
        self.btn_close_camera = QPushButton("Fermer Caméra")
        self.btn_capture.setEnabled(False)
        self.btn_close_camera.setEnabled(False)
        camera_btns.addWidget(self.btn_open_camera)
        camera_btns.addWidget(self.btn_capture)
        camera_btns.addWidget(self.btn_close_camera)
        right_layout.addLayout(camera_btns)

        self.info_label = QLabel("")
        self.info_label.setWordWrap(True)
        right_layout.addWidget(self.info_label)

        right_layout.addStretch()

        # ---------------------------
        # Liste des familles
        # ---------------------------
        self.families_list = QListWidget()
        right_layout.addWidget(QLabel("DB Familles :"))
        right_layout.addWidget(self.families_list)

        main_layout.addLayout(right_layout, stretch=1)

        # ---------------------------
        # Connexions
        # ---------------------------
        self.btn_open_camera.clicked.connect(self.start_camera)
        self.btn_capture.clicked.connect(self.capture_photo)
        self.btn_close_camera.clicked.connect(self.stop_camera)
        self.btn_clear.clicked.connect(self.clear_form)
        self.btn_save.clicked.connect(self.save_family)

        # Affichage initial
        self.update_families_list()

    # ---------------------------
    # Caméra
    # ---------------------------
    def start_camera(self):
        if self.cap is not None:
            return
        self.cap = cv2.VideoCapture(0, cv2.CAP_DSHOW)
        if not self.cap.isOpened():
            self.cap.release()
            self.cap = cv2.VideoCapture(0)
        if not self.cap.isOpened():
            QMessageBox.warning(self, "Caméra", "Impossible d'ouvrir la caméra.")
            self.cap = None
            return
        self.timer.start(30)
        self.btn_open_camera.setEnabled(False)
        self.btn_capture.setEnabled(True)
        self.btn_close_camera.setEnabled(True)
        self.info_label.setText("Caméra ouverte. Cliquez sur Capturer.")

    def _query_frame(self):
        if not self.cap:
            return
        ret, frame = self.cap.read()
        if not ret:
            return
        self.current_frame = frame
        try:
            frame_rgb = cv2.cvtColor(frame, cv2.COLOR_BGR2RGB)
            h, w, ch = frame_rgb.shape
            bytes_per_line = ch * w
            qimg = QImage(frame_rgb.data, w, h, bytes_per_line, QImage.Format.Format_RGB888)
            pix = QPixmap.fromImage(qimg)
            pix = pix.scaled(self.photo_label.width(), self.photo_label.height(),
                             Qt.AspectRatioMode.KeepAspectRatioByExpanding,
                             Qt.TransformationMode.SmoothTransformation)
            self.photo_label.setPixmap(pix)
        except Exception:
            pass

    def capture_photo(self):
        if self.current_frame is None:
            QMessageBox.information(self, "Capture", "Aucune image.")
            return
        ts = datetime.now().strftime("%Y%m%d_%H%M%S")
        fname = f"family_{ts}.png"
        out_dir = os.path.join(os.getcwd(), "database")
        os.makedirs(out_dir, exist_ok=True)
        path = os.path.join(out_dir, fname)
        cv2.imwrite(path, self.current_frame)
        self.last_photo_path = path
        frame_rgb = cv2.cvtColor(self.current_frame, cv2.COLOR_BGR2RGB)
        h, w, ch = frame_rgb.shape
        bytes_per_line = ch * w
        qimg = QImage(frame_rgb.data, w, h, bytes_per_line, QImage.Format.Format_RGB888)
        pix = QPixmap.fromImage(qimg).scaled(self.photo_label.width(),
                                             self.photo_label.height(),
                                             Qt.AspectRatioMode.KeepAspectRatioByExpanding,
                                             Qt.TransformationMode.SmoothTransformation)
        self.photo_label.setPixmap(pix)
        self.info_label.setText(f"Photo capturée : {os.path.basename(path)}")
        self.stop_camera()

    def stop_camera(self):
        if self.timer.isActive():
            self.timer.stop()
        if self.cap:
            try:
                self.cap.release()
            except Exception:
                pass
            self.cap = None
        self.btn_open_camera.setEnabled(True)
        self.btn_capture.setEnabled(False)
        self.btn_close_camera.setEnabled(False)
        self.info_label.setText("Caméra fermée.")

    # ---------------------------
    # Form utilities
    # ---------------------------
    def clear_form(self):
        self.input_id.clear()
        self.input_nom.clear()
        self.input_prenom.clear()
        self.input_adresse.clear()
        self.input_contact.clear()
        self.photo_label.setText("Aucune photo")
        self.last_photo_path = None
        self.info_label.setText("Formulaire effacé.")

    def save_family(self):
        data = {
            "id": self.input_id.text(),
            "nom_responsable": self.input_nom.text(),
            "prenom_responsable": self.input_prenom.text(),
            "adresse": self.input_adresse.text(),
            "contact": self.input_contact.text(),
            "photo": self.last_photo_path
        }
        try:
            conn = sqlite3.connect(DB_PATH)
            c = conn.cursor()
            c.execute("""
                INSERT OR REPLACE INTO families VALUES (:id, :nom_responsable, :prenom_responsable,
                                                        :adresse, :contact, :photo)
            """, data)
            conn.commit()
            conn.close()
            QMessageBox.information(self, "Succès", "Famille enregistrée avec succès.")
            self.clear_form()
            self.update_families_list()
        except Exception as e:
            QMessageBox.warning(self, "Erreur", f"Impossible d'enregistrer : {e}")

    def update_families_list(self):
        self.families_list.clear()
        conn = sqlite3.connect(DB_PATH)
        c = conn.cursor()
        c.execute("SELECT id, nom_responsable, prenom_responsable, adresse, contact FROM families ORDER BY id")
        for row in c.fetchall():
            item = QListWidgetItem(f"{row[0]} | {row[1]} {row[2]} | {row[3]} | {row[4]}")
            self.families_list.addItem(item)
        conn.close()
