# gui/change_password.py
import os
import sqlite3
from PyQt6.QtWidgets import (
    QWidget, QLabel, QLineEdit, QPushButton, QVBoxLayout,
    QSpacerItem, QSizePolicy, QMessageBox
)
from PyQt6.QtGui import QFont, QCursor
from PyQt6.QtCore import Qt

BASE_DIR = os.getcwd()
USERS_DB = os.path.join(BASE_DIR, "database", "users.db")


class ChangePasswordPage(QWidget):
    def __init__(self, username, parent=None):
        super().__init__(parent)
        self.username = username

        self.setWindowTitle("Modifier le mot de passe - RLM")
        self.setFixedSize(480, 620)

        # 🎨 STYLE PREMIUM bleu foncé
        self.setStyleSheet("""
            QWidget {
                background-color: #001a52;
                color: #ffffff;
                font-family: 'Segoe UI';
            }
            QLineEdit {
                padding: 12px;
                border-radius: 8px;
                border: 1px solid #415a77;
                background: #ffffff;
                color: #000000;
                font-size: 14px;
            }
            QPushButton {
                background-color: #00a7de;
                padding: 12px;
                border-radius: 8px;
                color: white;
                font-size: 16px;
                font-weight: bold;
            }
            QPushButton:hover {
                background-color: #007298;
            }
            QLabel#title {
                font-size: 26px;
                font-weight: bold;
                color: #ffffff;
            }
        """)

        layout = QVBoxLayout()
        layout.setContentsMargins(50, 20, 50, 20)
        layout.setSpacing(18)
        layout.setAlignment(Qt.AlignmentFlag.AlignTop)
        self.setLayout(layout)

        # TITRE
        title = QLabel("Modifier le mot de passe")
        title.setObjectName("title")
        title.setAlignment(Qt.AlignmentFlag.AlignCenter)
        layout.addWidget(title)

        # CHAMPS FORMULAIRE
        self.old_pwd = QLineEdit()
        self.old_pwd.setPlaceholderText("Ancien mot de passe")
        self.old_pwd.setEchoMode(QLineEdit.EchoMode.Password)
        layout.addWidget(self.old_pwd)

        self.new_pwd = QLineEdit()
        self.new_pwd.setPlaceholderText("Nouveau mot de passe")
        self.new_pwd.setEchoMode(QLineEdit.EchoMode.Password)
        layout.addWidget(self.new_pwd)

        self.confirm_pwd = QLineEdit()
        self.confirm_pwd.setPlaceholderText("Confirmer le mot de passe")
        self.confirm_pwd.setEchoMode(QLineEdit.EchoMode.Password)
        layout.addWidget(self.confirm_pwd)

        # VALIDATION
        self.save_btn = QPushButton("Modifier")
        self.save_btn.clicked.connect(self.update_password)
        layout.addWidget(self.save_btn)

        layout.addItem(QSpacerItem(20, 50, QSizePolicy.Policy.Minimum, QSizePolicy.Policy.Expanding))

        # FOOTER
        footer = QLabel("RLM SmartID-Tech © 2025 — DTech / D-Square")
        footer.setAlignment(Qt.AlignmentFlag.AlignCenter)
        footer.setStyleSheet("font-size: 10px; color: #a8b3c7;")
        layout.addWidget(footer)

    # -----------------------------------------
    #        MISE À JOUR DU MOT DE PASSE
    # -----------------------------------------
    def update_password(self):
        old = self.old_pwd.text().strip()
        new = self.new_pwd.text().strip()
        confirm = self.confirm_pwd.text().strip()

        if not old or not new or not confirm:
            QMessageBox.warning(self, "Erreur", "Veuillez remplir tous les champs.")
            return

        if new != confirm:
            QMessageBox.warning(self, "Erreur", "Les nouveaux mots de passe ne correspondent pas.")
            return

        if len(new) < 5:
            QMessageBox.warning(self, "Erreur", "Le mot de passe doit avoir au moins 5 caractères.")
            return

        # Vérification dans la DB
        conn = sqlite3.connect(USERS_DB)
        c = conn.cursor()

        c.execute("SELECT password FROM users WHERE username=?", (self.username,))
        row = c.fetchone()

        if not row:
            QMessageBox.critical(self, "Erreur", "Utilisateur introuvable.")
            conn.close()
            return

        old_password_db = row[0]

        if old != old_password_db:
            QMessageBox.warning(self, "Erreur", "Ancien mot de passe incorrect.")
            conn.close()
            return

        # Mise à jour
        c.execute("UPDATE users SET password=? WHERE username=?", (new, self.username))
        conn.commit()
        conn.close()

        QMessageBox.information(self, "Succès", "Mot de passe modifié avec succès !")
        self.close()
