# gui/admin_panel.py
import sys
import os
import sqlite3
from functools import partial
from datetime import datetime
import os
import shutil
import cv2
import pandas as pd
from PyQt6.QtWidgets import (
    QApplication, QWidget, QHBoxLayout, QVBoxLayout, QPushButton, QLabel,
    QLineEdit, QTableWidget, QTableWidgetItem, QMessageBox, QFileDialog,
    QProgressBar, QSizePolicy, QInputDialog, QProgressDialog, QDialog,
    QFormLayout, QDialogButtonBox
)
from PyQt6.QtCore import Qt, QObject, QEvent
from PyQt6.QtGui import QColor, QBrush

from gui.students_page import generate_id_card, generate_pdf
from PIL import Image
from PyQt6.QtWidgets import (
    QApplication, QWidget, QVBoxLayout, QHBoxLayout, QLabel, QPushButton,
    QLineEdit, QTextEdit, QTableWidget, QTableWidgetItem, QMessageBox,
    QFileDialog, QHeaderView, QInputDialog, QProgressBar, QSizePolicy  # <-- ajoute ceci
)


# ------------------------------
# Helper : crop/resize photo
# ------------------------------
def crop_and_resize_photo(photo_path, target_size=(300, 400)):
    try:
        img = Image.open(photo_path)
        width, height = img.size
        target_w, target_h = target_size
        aspect_ratio = width / height
        target_ratio = target_w / target_h

        if aspect_ratio > target_ratio:
            new_width = int(height * target_ratio)
            left = (width - new_width) // 2
            img = img.crop((left, 0, left + new_width, height))
        else:
            new_height = int(width / target_ratio)
            top = (height - new_height) // 2
            img = img.crop((0, top, width, top + new_height))

        img = img.resize(target_size, Image.Resampling.LANCZOS)
        img.save(photo_path)
        return photo_path
    except Exception as e:
        print(f"⚠️ Erreur rognage/redimensionnement photo {photo_path}: {e}")
        return photo_path

PHOTOS_DIR = os.path.join(os.getcwd(), "photos")
CARDS_DIR = os.path.join(os.getcwd(), "cards")
os.makedirs(PHOTOS_DIR, exist_ok=True)
os.makedirs(CARDS_DIR, exist_ok=True)


# ------------------------------
# Paths / DB files
# ------------------------------
BASE_DIR = os.getcwd()
DB_DIR = os.path.join(BASE_DIR, "database")
os.makedirs(DB_DIR, exist_ok=True)

USERS_DB = os.path.join(DB_DIR, "users.db")
STUDENTS_DB = os.path.join(DB_DIR, "students.db")
FAMILIES_DB = os.path.join(DB_DIR, "families.db")
NUTRITION_DB = os.path.join(DB_DIR, "nutrition.db")

CARDS_DIR = os.path.join(BASE_DIR, "cards")
PDFS_DIR = os.path.join(BASE_DIR, "pdfs")
PHOTOS_DIR = os.path.join(BASE_DIR, "PhotoCarte")

os.makedirs(CARDS_DIR, exist_ok=True)
os.makedirs(PDFS_DIR, exist_ok=True)
os.makedirs(PHOTOS_DIR, exist_ok=True)

# ------------------------------
# Event filter pour détecter double-clic sur QPushButton
# ------------------------------
class ButtonDoubleClickFilter(QObject):
    def __init__(self, callback_doubleclick):
        super().__init__()
        self.callback_doubleclick = callback_doubleclick

    def eventFilter(self, obj, event):
        if event.type() == QEvent.Type.MouseButtonDblClick:
            try:
                self.callback_doubleclick(obj)
            except Exception as e:
                print("DoubleClick callback error:", e)
            return True
        return False

# ------------------------------
# AdminPanel
# ------------------------------
class AdminPanel(QWidget):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("Admin Panel - Redeeming Love Ministries")
        self.setMinimumSize(1200, 700)
        # Styles : menu bleu foncé, selected vert, logout rouge
        self.setStyleSheet("""
            QWidget { background-color: #1e1e1e; color: #f0f0f0; font-family: 'Segoe UI'; }
            QPushButton { background-color: #2d2d2d; color: #f0f0f0; padding: 10px; border-radius: 6px; }
            QPushButton:hover { background-color: #3a3a3a; }
            QPushButton.menuBtn { background-color: #0b3d91; color: white; font-weight: bold; }
            QPushButton.menuBtn[selected="true"] { background-color: #27ae60; color: white; font-weight: bold; }
            QPushButton.logoutBtn { background-color: #e74c3c; color: white; font-weight: bold; }
            QLineEdit { padding: 6px; border: 1px solid #555; border-radius: 6px; background-color: #2b2b2b; color: #f0f0f0; }
            QHeaderView::section { background-color: #444; color: white; padding: 6px; border: 1px solid #555; }
            QTableWidget { gridline-color: #555; background-color: #252525; alternate-background-color: #2d2d2d; color: #f0f0f0; }
            QLabel#title { font-size: 20pt; font-weight: bold; color: #ffcc00; padding:8px; }
        """)

        # Ensure DB
        self.ensure_users_db()
        self.ensure_students_db()
        self.ensure_families_db()
        self.ensure_nutrition_db()

        # -------------------
        # Layout principal
        # -------------------
        main_layout = QHBoxLayout(self)
        menu_layout = QVBoxLayout()
        menu_layout.setSpacing(10)

        title = QLabel("ADMIN PANEL")
        title.setObjectName("title")
        title.setAlignment(Qt.AlignmentFlag.AlignCenter)
        menu_layout.addWidget(title)

        # Menu buttons (créés une fois)
        self.menu_buttons = {}
        from PyQt6.QtWidgets import QSizePolicy
        def mk_menu_btn(text, objname="menuBtn"):
            b = QPushButton(text)
            b.setProperty("class", objname)
            b.setObjectName(text)  # utile pour debug
            b.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Fixed)
            b.setProperty("selected", False)
            b.setStyleSheet("")  # style global géré par stylesheet ci-dessus
            return b

        self.btn_dashboard = mk_menu_btn("📊 Dashboard")
        self.btn_users = mk_menu_btn("👥 Utilisateurs")
        self.btn_students = mk_menu_btn("🎓 BD Étudiants")
        self.btn_families = mk_menu_btn("🏠 BD Familles")
        self.btn_nutrition = mk_menu_btn("🥗 BD Nutrition")
        self.btn_import = mk_menu_btn("📥 Importer fichier Excel")
        self.btn_settings = mk_menu_btn("⚙️ Paramètres")
        self.btn_import_old_cards = QPushButton("Importer anciennes cartes")
        self.btn_import_old_cards.setStyleSheet("background-color:#3498db;color:white;font-weight:bold;")
        self.btn_import_old_cards.clicked.connect(self.import_old_cards_and_generate_new)
        menu_layout.addWidget(self.btn_import_old_cards)  # ajoute-le au layout du menu latéral
        self.btn_logout = mk_menu_btn("🔒 Déconnexion", objname="logoutBtn")
        self.btn_logout.setProperty("class", "logoutBtn")

        # store in dictionary (optional)
        self.menu_buttons = {
            "dashboard": self.btn_dashboard,
            "users": self.btn_users,
            "students": self.btn_students,
            "families": self.btn_families,
            "nutrition": self.btn_nutrition,
            "import": self.btn_import,
            "settings": self.btn_settings,
            "logout": self.btn_logout
        }

        # Add to menu layout
        for btn in [self.btn_dashboard, self.btn_users, self.btn_students,
                    self.btn_families, self.btn_nutrition, self.btn_import, self.btn_settings,
                    self.btn_logout]:
            # appliquer style class pour stylesheet
            cls = btn.property("class")
            if cls == "logoutBtn":
                btn.setProperty("class", "logoutBtn")
            else:
                btn.setProperty("class", "menuBtn")
            menu_layout.addWidget(btn)
        menu_layout.addStretch()
        main_layout.addLayout(menu_layout, 1)

        # Contenu principal (column right)
        self.content_layout = QVBoxLayout()
        main_layout.addLayout(self.content_layout, 4)

        # Barre recherche + table
        self.search_input = QLineEdit()
        self.search_input.setPlaceholderText("Rechercher par Matricule / Nom / Date...")
        self.content_layout.addWidget(self.search_input)

        self.table = QTableWidget()
        self.content_layout.addWidget(self.table)

        # Status + action buttons (créés une seule fois)
        self.status_layout = QHBoxLayout()
        self.status_buttons = []
        status_defs = [
            ("Abandon", "#3498db"), ("Décédé", "#e74c3c"), ("Marié(e)", "#9b59b6"),
            ("Engrossée", "#f1c40f"), ("Fin suivi", "#2ecc71"), ("Réaffecté", "#e67e22")
        ]
        for name, color in status_defs:
            btn = QPushButton(name)
            btn.setStyleSheet(f"background-color:{color}; color:white; font-weight:bold;")
            btn.clicked.connect(partial(self.set_student_status, name, color))
            btn.setVisible(False)
            btn.setEnabled(False)
            self.status_layout.addWidget(btn)
            self.status_buttons.append(btn)

        # Bouton Supprimer étudiant (hors table)
        self.btn_delete_student = QPushButton("Supprimer étudiant sélectionné")
        self.btn_delete_student.setStyleSheet("background-color:#e74c3c; color:white; font-weight:bold;")
        self.btn_delete_student.clicked.connect(self.delete_selected_student)
        self.btn_delete_student.setVisible(False)
        self.btn_delete_student.setEnabled(False)
        self.status_layout.addWidget(self.btn_delete_student)

        # Buttons Utilisateurs (hors table)
        self.btn_add_user = QPushButton("Ajouter")
        self.btn_edit_user = QPushButton("Modifier")
        self.btn_delete_user = QPushButton("Supprimer")
        self.btn_suspend_user = QPushButton("Suspendre")
        for b in [self.btn_add_user, self.btn_edit_user, self.btn_delete_user, self.btn_suspend_user]:
            b.setStyleSheet("background-color:#0b3d91; color:white; font-weight:bold;")
            b.setVisible(False)
            b.setEnabled(False)
            self.status_layout.addWidget(b)

        # connecter actions utilisateurs
        self.btn_add_user.clicked.connect(self.add_user)
        self.btn_edit_user.clicked.connect(self.edit_user_password)
        self.btn_delete_user.clicked.connect(self.delete_user)
        self.btn_suspend_user.clicked.connect(self.suspend_user)

        self.content_layout.addLayout(self.status_layout)

        # Progress bar (import)
        self.progress = QProgressBar()
        self.progress.setVisible(False)
        self.content_layout.addWidget(self.progress)

        # Dashboard panel (conteneur des boutons de comptage) - créé une seule fois
        from PyQt6.QtWidgets import QFrame, QSizePolicy
        self.dashboard_panel = QFrame()
        self.dashboard_panel_layout = QVBoxLayout(self.dashboard_panel)
        self.dashboard_panel.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Expanding)
        # style des boutons comptage sera appliqué lorsque montrés
        self.dashboard_buttons = []  # liste de QPushButton (créés en show_dashboard)
        # on ajoute le panel mais on le cache par défaut
        self.content_layout.addWidget(self.dashboard_panel)
        self.dashboard_panel.setVisible(False)

        # -------------------
        # Connections (menu)
        # -------------------
        self.btn_dashboard.clicked.connect(lambda: self.menu_navigate("dashboard"))
        self.btn_users.clicked.connect(lambda: self.menu_navigate("users"))
        self.btn_students.clicked.connect(lambda: self.menu_navigate("students"))
        self.btn_families.clicked.connect(lambda: self.menu_navigate("families"))
        self.btn_nutrition.clicked.connect(lambda: self.menu_navigate("nutrition"))
        self.btn_import.clicked.connect(lambda: self.menu_navigate("import"))
        self.btn_settings.clicked.connect(lambda: self.menu_navigate("settings"))
        self.btn_logout.clicked.connect(self.logout)

        self.search_input.textChanged.connect(self.filter_table_live)
        self.table.itemSelectionChanged.connect(self.on_row_selection_change)
        self.table.itemDoubleClicked.connect(self.on_table_double_click)

        # mise en place initiale : dashboard visible
        self.current_table = None
        self.set_active_menu(self.btn_dashboard)
        self.load_table("dashboard")

    # ------------------- DB -------------------
    def ensure_users_db(self):
        conn = sqlite3.connect(USERS_DB); c = conn.cursor()
        c.execute("""CREATE TABLE IF NOT EXISTS users (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            username TEXT UNIQUE,
            password TEXT,
            role TEXT,
            status TEXT DEFAULT 'Actif'
        )""")

    def ensure_students_db(self):
        conn = sqlite3.connect(STUDENTS_DB); c = conn.cursor()
        c.execute("""CREATE TABLE IF NOT EXISTS students (
            matricule TEXT PRIMARY KEY,
            nom TEXT, postnom TEXT, prenom TEXT,
            pere TEXT, mere TEXT, naissance TEXT,
            sexe TEXT, institution TEXT, classe TEXT,
            resultat TEXT, mention TEXT, photo TEXT,
            statut TEXT DEFAULT 'actif',
            owner TEXT, date_enreg TEXT DEFAULT CURRENT_TIMESTAMP
        )""")
        conn.commit(); conn.close()

    def ensure_families_db(self):
        conn = sqlite3.connect(FAMILIES_DB); c = conn.cursor()
        c.execute("""CREATE TABLE IF NOT EXISTS families (
            matricule TEXT PRIMARY KEY,
            nom_famille TEXT, responsable TEXT, zone TEXT,
            adresse TEXT, nb_enfants INTEGER, categorie TEXT,
            date_debut_suivi TEXT, date_fin_suivi TEXT, photo TEXT,
            owner TEXT, date_enreg TEXT DEFAULT CURRENT_TIMESTAMP
        )""")
        conn.commit(); conn.close()

    def ensure_nutrition_db(self):
        conn = sqlite3.connect(NUTRITION_DB); c = conn.cursor()
        c.execute("""CREATE TABLE IF NOT EXISTS nutrition (
            num_dossier TEXT PRIMARY KEY,
            nom TEXT, sexe TEXT, date_naissance TEXT,
            nom_parent TEXT, adresse TEXT, date_enreg TEXT,
            poids REAL, pb REAL, scorez REAL,
            oedemes TEXT, appetit_bon TEXT, autres TEXT, photo TEXT
        )""")
        conn.commit(); conn.close()

    # ------------------- Menu management -------------------
    def set_active_menu(self, btn):
        # reset all menu buttons to default (bleu foncé)
        for key, b in self.menu_buttons.items():
            if b is None: continue
            # mark selected property
            b.setProperty("selected", False)
            b.style().unpolish(b)
            b.style().polish(b)
        # set active
        if btn:
            btn.setProperty("selected", True)
            btn.style().unpolish(btn)
            btn.style().polish(btn)

    def menu_navigate(self, key):
        # map import/settings to correct behavior
        if key == "import":
            # open import dialog flow
            self.import_excel()
            # keep menu selection at import
            self.set_active_menu(self.btn_import)
            return
        if key == "settings":
            self.show_settings()
            self.set_active_menu(self.btn_settings)
            return
        if key == "logout":
            self.logout()
            return

        # normal table views
        btn = self.menu_buttons.get(key)
        self.set_active_menu(btn)
        self.load_table(key)

    # ------------------- Logout -------------------
    def logout(self):
        # tentatively assume login_page exists and handles being shown
        try:
            from gui.login_page import LoginPage
            self.close()
            self.login_window = LoginPage()
            self.login_window.show()
        except Exception as e:
            QMessageBox.information(self, "Déconnexion", "Déconnexion (simulée). Fermez la fenêtre si vous voulez quitter.")
            print("Logout fallback:", e)

    # ------------------- Dashboard -------------------
    def show_dashboard(self):
        # clear table
        self.table.clear(); self.table.setRowCount(0); self.table.setColumnCount(0)
        # hide actions
        for b in self.status_buttons: b.setVisible(False)
        self.btn_delete_student.setVisible(False)
        for b in [self.btn_add_user, self.btn_edit_user, self.btn_delete_user, self.btn_suspend_user]:
            b.setVisible(False)

        # compute counts
        self.dashboard_counts = {
            "Users": self.count_rows(USERS_DB),
            "Students": self.count_rows(STUDENTS_DB),
            "Families": self.count_rows(FAMILIES_DB),
            "Nutrition": self.count_rows(NUTRITION_DB)
        }

        # Populate dashboard_panel only once, do not duplicate
        # Clear previous children in panel layout
        for i in reversed(range(self.dashboard_panel_layout.count())):
            w = self.dashboard_panel_layout.itemAt(i).widget()
            if w:
                w.setParent(None)

        self.dashboard_buttons = []
        filter_btn_style = "background-color:#0b3d91; color:white; font-weight:bold; padding:16px; font-size:14px; border-radius:8px;"
        for key, val in self.dashboard_counts.items():
            btn = QPushButton(f"{key}\n{val}")
            btn.setProperty("class", "menuBtn")
            btn.setStyleSheet(filter_btn_style)
            btn.setSizePolicy(QSizePolicy.Policy.Expanding, QSizePolicy.Policy.Fixed)
            # click shows the list (single click loads) ; double click also loads list (redundant but explicit)
            btn.clicked.connect(lambda checked, k=key.lower(): self.load_table(k))
            # install double-click filter to specifically call load_table
            btn.installEventFilter(ButtonDoubleClickFilter(lambda b: self.load_table(key.lower())))
            self.dashboard_panel_layout.addWidget(btn)
            self.dashboard_buttons.append(btn)

        # show the panel
        self.dashboard_panel.setVisible(True)

    def count_rows(self, db_file):
        try:
            conn = sqlite3.connect(db_file); c = conn.cursor()
            table_name_map = {
                USERS_DB: "users",
                STUDENTS_DB: "students",
                FAMILIES_DB: "families",
                NUTRITION_DB: "nutrition"
            }
            table = table_name_map.get(db_file)
            if not table:
                # allow passing table names directly
                if isinstance(db_file, str) and os.path.exists(db_file):
                    table = os.path.splitext(os.path.basename(db_file))[0]
                else:
                    return 0
            c.execute(f"SELECT COUNT(*) FROM {table}")
            res = c.fetchone()[0]
            conn.close()
            return res
        except Exception:
            return 0
    # ---------------------- Gestion Utilisateurs ----------------------
    def add_user(self):
        username, ok = QInputDialog.getText(self, "Nouvel utilisateur", "Nom d'utilisateur :")
        if not ok or not username: return
        password, ok = QInputDialog.getText(self, "Mot de passe", "Entrer mot de passe :")
        if not ok or not password: return
        role, ok = QInputDialog.getItem(self, "Rôle", "Choisir rôle :", ["admin", "user"], 0, False)
        if not ok: return

        conn = sqlite3.connect(USERS_DB)
        c = conn.cursor()
        try:
            c.execute("INSERT INTO users (username, password, role, status) VALUES (?, ?, ?, ?)",
                      (username, password, role, "Actif"))
            conn.commit()
            QMessageBox.information(self, "Succès", f"Utilisateur {username} ajouté.")
        except Exception as e:
            QMessageBox.warning(self, "Erreur", f"Impossible d'ajouter : {e}")
        finally:
            conn.close()
            self.load_table("users")

    def edit_user_password(self):
        row = self.table.currentRow()
        if row < 0:
            QMessageBox.warning(self, "Avertissement", "Veuillez sélectionner un utilisateur.")
            return
        username = self.table.item(row, 0).text()
        new_pass, ok = QInputDialog.getText(self, "Modifier mot de passe", f"Nouveau mot de passe pour {username} :")
        if not ok or not new_pass: return

        conn = sqlite3.connect(USERS_DB)
        c = conn.cursor()
        try:
            c.execute("UPDATE users SET password=? WHERE username=?", (new_pass, username))
            conn.commit()
            QMessageBox.information(self, "Succès", f"Mot de passe de {username} modifié.")
        except Exception as e:
            QMessageBox.warning(self, "Erreur", f"Échec modification : {e}")
        finally:
            conn.close()
            self.load_table("users")

    def edit_user_username(self):
        row = self.table.currentRow()
        if row < 0:
            QMessageBox.warning(self, "Avertissement", "Veuillez sélectionner un utilisateur.")
            return
        old_username = self.table.item(row, 0).text()
        new_username, ok = QInputDialog.getText(self, "Modifier nom", f"Nouveau nom pour {old_username} :")
        if not ok or not new_username: return

        conn = sqlite3.connect(USERS_DB)
        c = conn.cursor()
        try:
            c.execute("UPDATE users SET username=? WHERE username=?", (new_username, old_username))
            conn.commit()
            QMessageBox.information(self, "Succès", f"Nom d'utilisateur changé en {new_username}.")
        except Exception as e:
            QMessageBox.warning(self, "Erreur", f"Échec changement : {e}")
        finally:
            conn.close()
            self.load_table("users")

    def suspend_user(self):
        row = self.table.currentRow()
        if row < 0:
            QMessageBox.warning(self, "Avertissement", "Veuillez sélectionner un utilisateur.")
            return
        username = self.table.item(row, 0).text()

        conn = sqlite3.connect(USERS_DB)
        c = conn.cursor()
        try:
            c.execute("UPDATE users SET status=? WHERE username=?", ("Suspendu", username))
            conn.commit()
            QMessageBox.information(self, "Suspendu", f"L'utilisateur {username} est suspendu.")
        except Exception as e:
            QMessageBox.warning(self, "Erreur", f"Impossible de suspendre : {e}")
        finally:
            conn.close()
            self.load_table("users")

    def delete_user(self):
        row = self.table.currentRow()
        if row < 0:
            QMessageBox.warning(self, "Avertissement", "Veuillez sélectionner un utilisateur.")
            return
        username = self.table.item(row, 0).text()
        confirm = QMessageBox.question(self, "Confirmer", f"Supprimer définitivement {username} ?")
        if confirm != QMessageBox.StandardButton.Yes: return

        conn = sqlite3.connect(USERS_DB)
        c = conn.cursor()
        try:
            c.execute("DELETE FROM users WHERE username=?", (username,))
            conn.commit()
            QMessageBox.information(self, "Supprimé", f"{username} a été supprimé.")
        except Exception as e:
            QMessageBox.warning(self, "Erreur", f"Échec suppression : {e}")
        finally:
            conn.close()
            self.load_table("users")

    # ------------------- Load table -------------------
    def load_table(self, table_name):
        # When switching away from dashboard, hide dashboard panel
        if table_name != "dashboard":
            self.dashboard_panel.setVisible(False)

        # set current table and show/hide actions accordingly
        self.current_table = table_name
        show_actions = table_name == "students"
        for b in self.status_buttons:
            b.setVisible(show_actions)
            b.setEnabled(False)
        self.btn_delete_student.setVisible(show_actions)
        self.btn_delete_student.setEnabled(False)

        # Users action buttons visible only in users view
        show_users_btns = table_name == "users"
        for b in [self.btn_edit_user, self.btn_delete_user, self.btn_suspend_user]:
            b.setVisible(show_users_btns)
            b.setEnabled(False)  # restent désactivés jusqu'à sélection
        # Le bouton Ajouter doit toujours être visible et activé
        self.btn_add_user.setVisible(show_users_btns)
        self.btn_add_user.setEnabled(show_users_btns)  # activé dès que vue "users"

        # clear table
        self.table.clear()
        self.table.setRowCount(0)

        if table_name == "dashboard":
            # activate dashboard menu appearance
            self.dashboard_panel.setVisible(True)
            self.show_dashboard()
            return

        # map for dbs and headers
        db_map = {"users": USERS_DB, "students": STUDENTS_DB, "families": FAMILIES_DB, "nutrition": NUTRITION_DB}
        headers_map = {
            "users": ["Number", "Users", "Password"],
            "students": ["Matricule","Nom","Postnom","Prénom","Père","Mère","Naissance","Sexe",
                         "Institution","Classe","Résultat","Mention","Photo","Statut","Owner","Date Enreg."],
            "families": ["Matricule","Nom Famille","Responsable","Zone","Adresse","Nb Enfants",
                         "Catégorie","Date Debut Suivi","Date Fin Suivi","Photo","Owner","Date Enreg."],
            "nutrition": ["Num Dossier","Nom","Sexe","Date Naissance","Nom Parent","Adresse",
                          "Date Enreg.","Poids","PB","ScoreZ","Oedèmes","Appétit Bon","Autres","Photo"]
        }

        db_path = db_map.get(table_name)
        headers = headers_map.get(table_name, [])

        if not db_path:
            QMessageBox.warning(self, "Erreur", f"DB non trouvée pour {table_name}")
            return

        # load rows
        conn = sqlite3.connect(db_path); c = conn.cursor()
        try:
            rows = c.execute(f"SELECT * FROM {table_name}").fetchall()
        except Exception as e:
            QMessageBox.warning(self, "Erreur DB", f"Impossible de charger : {e}")
            conn.close(); return
        conn.close()

        # fill table
        self.table.setColumnCount(len(headers))
        self.table.setHorizontalHeaderLabels(headers)
        for i, row in enumerate(rows):
            self.table.insertRow(i)
            for j, val in enumerate(row):
                self.table.setItem(i, j, QTableWidgetItem("" if val is None else str(val)))

    # ------------------- Filter live -------------------
    def filter_table_live(self, text):
        text = text.strip().lower()
        for r in range(self.table.rowCount()):
            show = any(self.table.item(r, c) and text in self.table.item(r, c).text().lower()
                       for c in range(self.table.columnCount()))
            self.table.setRowHidden(r, not show)

    # ------------------- Row selection -------------------
    def on_row_selection_change(self):
        row = self.table.currentRow()
        if row < 0:
            return
        if self.current_table == "students":
            for b in self.status_buttons:
                b.setEnabled(True)
            self.btn_delete_student.setEnabled(True)
        elif self.current_table == "users":
            for b in [self.btn_add_user, self.btn_edit_user, self.btn_delete_user, self.btn_suspend_user]:
                b.setEnabled(True)

    # ------------------- Double click table -------------------
    def on_table_double_click(self, item):
        # user requested double-click on table rows: can be extended to open detail panels
        if self.current_table in ["users", "students", "families", "nutrition"]:
            QMessageBox.information(self, "Info", f"Double-clic : affichage complet déjà activé pour {self.current_table}.")
            # Already the table is the full list — we could implement details view here.

    # ------------------- User actions (Ajouter / Modifier / Supprimer / Suspendre) -------------------
    def action_add_user(self):
        # modal dialog for username/password/role
        dlg = QDialog(self)
        dlg.setWindowTitle("Ajouter utilisateur")
        layout = QFormLayout(dlg)
        inp_user = QLineEdit(); inp_pass = QLineEdit(); inp_pass.setEchoMode(QLineEdit.EchoMode.Password)
        inp_role = QLineEdit(); inp_role.setPlaceholderText("ex: admin / user")
        layout.addRow("Username:", inp_user)
        layout.addRow("Password:", inp_pass)
        layout.addRow("Role:", inp_role)
        buttons = QDialogButtonBox(QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel)
        layout.addWidget(buttons)
        buttons.accepted.connect(dlg.accept); buttons.rejected.connect(dlg.reject)
        if dlg.exec() != QDialog.DialogCode.Accepted:
            return
        username = inp_user.text().strip()
        password = inp_pass.text().strip()
        role = inp_role.text().strip() or "user"
        if not username or not password:
            QMessageBox.warning(self, "Erreur", "Username et Password requis.")
            return
        try:
            conn = sqlite3.connect(USERS_DB); c = conn.cursor()
            c.execute("INSERT INTO users (username,password,role) VALUES (?,?,?)", (username, password, role))
            conn.commit(); conn.close()
            QMessageBox.information(self, "OK", f"Utilisateur {username} ajouté.")
            # refresh users table
            self.load_table("users")
        except Exception as e:
            QMessageBox.warning(self, "Erreur DB", f"Impossible d'ajouter l'utilisateur : {e}")

    def action_edit_user(self):
        # edit username or password of selected user
        row = self.table.currentRow()
        if row < 0:
            QMessageBox.warning(self, "Erreur", "Sélectionner un utilisateur.")
            return
        current_username_item = self.table.item(row, 0)
        if not current_username_item:
            QMessageBox.warning(self, "Erreur", "Sélection invalide.")
            return
        current_username = current_username_item.text()

        dlg = QDialog(self)
        dlg.setWindowTitle(f"Modifier utilisateur : {current_username}")
        layout = QFormLayout(dlg)
        inp_user = QLineEdit(current_username)
        inp_pass = QLineEdit()
        inp_pass.setEchoMode(QLineEdit.EchoMode.Password)
        inp_role = QLineEdit(self.table.item(row, 2).text() if self.table.item(row, 2) else "")
        layout.addRow("Username:", inp_user)
        layout.addRow("Password (laisser vide si inchangé):", inp_pass)
        layout.addRow("Role:", inp_role)
        buttons = QDialogButtonBox(QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel)
        layout.addWidget(buttons)
        buttons.accepted.connect(dlg.accept); buttons.rejected.connect(dlg.reject)
        if dlg.exec() != QDialog.DialogCode.Accepted:
            return
        new_username = inp_user.text().strip()
        new_password = inp_pass.text().strip()
        new_role = inp_role.text().strip() or "user"
        try:
            conn = sqlite3.connect(USERS_DB); c = conn.cursor()
            if new_password:
                c.execute("UPDATE users SET username=?, password=?, role=? WHERE username=?", (new_username, new_password, new_role, current_username))
            else:
                c.execute("UPDATE users SET username=?, role=? WHERE username=?", (new_username, new_role, current_username))
            conn.commit(); conn.close()
            QMessageBox.information(self, "OK", f"Utilisateur mis à jour.")
            self.load_table("users")
        except Exception as e:
            QMessageBox.warning(self, "Erreur DB", f"Impossible de modifier : {e}")

    def action_delete_user(self):
        row = self.table.currentRow()
        if row < 0:
            QMessageBox.warning(self, "Erreur", "Sélectionner un utilisateur.")
            return
        username_item = self.table.item(row, 0)
        if not username_item:
            return
        username = username_item.text()
        confirm = QMessageBox.question(self, "Confirmer", f"Supprimer l'utilisateur {username} ?", QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No)
        if confirm != QMessageBox.StandardButton.Yes:
            return
        try:
            conn = sqlite3.connect(USERS_DB); c = conn.cursor()
            c.execute("DELETE FROM users WHERE username=?", (username,))
            conn.commit(); conn.close()
            QMessageBox.information(self, "OK", "Utilisateur supprimé.")
            self.load_table("users")
        except Exception as e:
            QMessageBox.warning(self, "Erreur DB", f"Impossible de supprimer : {e}")

    def action_suspend_user(self):
        row = self.table.currentRow()
        if row < 0:
            QMessageBox.warning(self, "Erreur", "Sélectionner un utilisateur.")
            return
        username_item = self.table.item(row, 0)
        if not username_item:
            return
        username = username_item.text()
        confirm = QMessageBox.question(self, "Suspendre", f"Suspendre l'utilisateur {username} ? (role → suspended)", QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No)
        if confirm != QMessageBox.StandardButton.Yes:
            return
        try:
            conn = sqlite3.connect(USERS_DB); c = conn.cursor()
            c.execute("UPDATE users SET role=? WHERE username=?", ("suspended", username))
            conn.commit(); conn.close()
            QMessageBox.information(self, "OK", "Utilisateur suspendu (role = suspended).")
            self.load_table("users")
        except Exception as e:
            QMessageBox.warning(self, "Erreur DB", f"Impossible de suspendre : {e}")

    # ------------------- Set status student -------------------
    def set_student_status(self, status_label, color_hex):
        if self.current_table != "students":
            QMessageBox.warning(self, "Erreur", "Basculer la table sur 'students' d'abord."); return
        row = self.table.currentRow()
        if row < 0:
            QMessageBox.warning(self, "Erreur", "Sélectionner une ligne d'étudiant."); return
        matricule_item = self.table.item(row, 0)
        if not matricule_item:
            return
        matricule = matricule_item.text()
        try:
            conn = sqlite3.connect(STUDENTS_DB); c = conn.cursor()
            c.execute("UPDATE students SET statut=? WHERE matricule=?", (status_label, matricule))
            conn.commit(); conn.close()
        except Exception as e:
            QMessageBox.warning(self, "Erreur DB", f"Impossible de mettre à jour statut : {e}"); return

        color = QColor(color_hex)
        for col in range(self.table.columnCount()):
            it = self.table.item(row, col)
            if it:
                it.setBackground(QBrush(color))
        QMessageBox.information(self, "Statut", f"Statut {matricule} → {status_label}")

    # ------------------- Supprimer étudiant sélectionné -------------------
    def delete_selected_student(self):
        if self.current_table != "students":
            return
        row = self.table.currentRow()
        if row < 0:
            QMessageBox.warning(self, "Erreur", "Sélectionner une ligne d'étudiant."); return
        matricule_item = self.table.item(row, 0)
        if not matricule_item:
            return
        matricule = matricule_item.text()
        confirm = QMessageBox.question(self, "Confirmer suppression", f"Supprimer l'étudiant {matricule} ?",
                                       QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No)
        if confirm != QMessageBox.StandardButton.Yes:
            return
        try:
            conn = sqlite3.connect(STUDENTS_DB); c = conn.cursor()
            c.execute("DELETE FROM students WHERE matricule=?", (matricule,))
            conn.commit(); conn.close()
            self.table.removeRow(row)
            QMessageBox.information(self, "Supprimé", f"Étudiant {matricule} supprimé.")
        except Exception as e:
            QMessageBox.warning(self, "Erreur DB", f"Impossible de supprimer : {e}")

    # ------------------- Import Excel -------------------
    def import_excel(self):
        # choix table
        options = ["students", "families", "nutrition"]
        default_index = 0
        table_choice, ok = QInputDialog.getItem(self, "Importer", "Importer vers :", options, default_index, False)
        if not ok:
            return
        table_name = table_choice

        # choix utilisateur (owner)
        conn = sqlite3.connect(USERS_DB); c = conn.cursor()
        users = [row[0] for row in c.execute("SELECT username FROM users").fetchall()]
        conn.close()
        if not users:
            QMessageBox.warning(self, "Erreur", "Pas d'utilisateur trouvé. Créez un utilisateur d'abord.")
            return
        user_choice, ok = QInputDialog.getItem(self, "Utilisateur", "Associer les données à l'utilisateur :", users, 0, False)
        if not ok:
            return
        owner = user_choice

        # choisir fichier
        file_path, _ = QFileDialog.getOpenFileName(self, "Importer fichier Excel", "", "Excel Files (*.xlsx *.xls)")
        if not file_path:
            return
        try:
            df = pd.read_excel(file_path, dtype=str).fillna("")
        except Exception as e:
            QMessageBox.warning(self, "Erreur", f"Impossible de lire le fichier Excel : {e}")
            return

        # Progress dialog + import
        total = len(df)
        progress = QProgressDialog("Importation en cours...", "Annuler", 0, max(1, total), self)
        progress.setWindowTitle("Import Excel")
        progress.setWindowModality(Qt.WindowModality.WindowModal)
        progress.show()
        self.progress.setVisible(True)
        self.progress.setMaximum(total); self.progress.setValue(0)

        self._import_to_db(df, table_name, owner, progress)
        self.progress.setVisible(False)
        # recharge la vue active
        self.load_table(table_name)

    def _import_to_db(self, df, table_name, owner, progress_dialog=None):
        # mapping identique aux versions précédentes
        if table_name == "students":
            mapping = {
                "Matricule": "matricule", "Nom": "nom", "Post-Nom": "postnom", "Prénom": "prenom",
                "Nom du Père": "pere", "Nom de la Mère": "mere", "Naissance": "naissance", "Sexe": "sexe",
                "Institution": "institution", "Classe": "classe", "Resultat": "resultat", "Mention": "mention",
                "PHOTO": "photo"
            }
            db_path, db_table = STUDENTS_DB, "students"
        elif table_name == "families":
            mapping = {
                "Matricule":"matricule","Nom de la Famille":"nom_famille","Responsable":"responsable",
                "Zone":"zone","Adresse":"adresse","Nombre D'Enfants":"nb_enfants","Categorie":"categorie",
                "Date début de suivi":"date_debut_suivi","Date fin de suivi":"date_fin_suivi","PHOTO":"photo"
            }
            db_path, db_table = FAMILIES_DB, "families"
        else:
            mapping = {
                "Nom":"nom","Sexe":"sexe","Date de Naissance":"date_naissance","Nom Parent":"nom_parent",
                "Adresse":"adresse","Date d'Enreg.":"date_enreg","Num Dossier":"num_dossier","Poids":"poids",
                "PB":"pb","ScoreZ":"scorez","Oedèmes":"oedemes","Apétit Bon":"appetit_bon","Autres":"autres","Photo":"photo"
            }
            db_path, db_table = NUTRITION_DB, "nutrition"

        total = len(df)
        conn = sqlite3.connect(db_path); c = conn.cursor()
        inserted = 0

        for i, row in df.iterrows():
            if progress_dialog and progress_dialog.wasCanceled():
                break
            if progress_dialog:
                progress_dialog.setValue(i)
            self.progress.setValue(i)

            rec = {}
            for excel_col, db_col in mapping.items():
                found = next((orig for orig in df.columns if orig.strip().lower() == excel_col.strip().lower()), None)
                val = str(row[found]).strip() if found else ""
                if db_col == "photo":
                    photo_name = os.path.basename(val)
                    full_path = os.path.join(PHOTOS_DIR, photo_name)
                    if photo_name and os.path.exists(full_path):
                        crop_and_resize_photo(full_path, (300, 400))
                        rec["photo"] = photo_name
                    else:
                        rec["photo"] = ""
                else:
                    rec[db_col] = val

            rec["owner"] = owner
            rec.setdefault("date_enreg", datetime.now().strftime("%Y-%m-%d %H:%M:%S"))
            if db_table == "students":
                rec.setdefault("statut", "actif")

            c.execute(f"PRAGMA table_info({db_table})")
            existing_cols = [r[1] for r in c.fetchall()]
            insert_cols = [col for col in existing_cols if col in rec]
            if not insert_cols:
                continue
            placeholders = ",".join(["?"]*len(insert_cols))
            cols_sql = ",".join(insert_cols)
            values = [rec[col] for col in insert_cols]
            try:
                c.execute(f"INSERT OR REPLACE INTO {db_table} ({cols_sql}) VALUES ({placeholders})", tuple(values))
            except Exception as e:
                print(f"Erreur insertion ligne {i}: {e}")
            inserted += 1

            # génération ID/PDF pour students
            if db_table == "students":
                data_for_card = {k: rec.get(k, "") for k in ["matricule","nom","postnom","prenom","pere","mere","naissance","sexe","institution","classe"]}
                data_for_card["photo"] = os.path.join(PHOTOS_DIR, rec.get("photo","")) if rec.get("photo") else ""
                try:
                    generate_id_card(data_for_card)
                    generate_pdf(data_for_card)
                except Exception as e:
                    print("Erreur génération ID/PDF:", e)

        conn.commit(); conn.close()
        if progress_dialog:
            progress_dialog.setValue(total)
            progress_dialog.close()
        self.progress.setValue(total)
        QMessageBox.information(self, "Terminé", f"Import terminé. Lignes traitées: {total} | insérées: {inserted}")

    def import_old_cards_and_generate_new(self):
        import cv2, qrcode, os
        import numpy as np
        from datetime import datetime
        from PyQt6.QtWidgets import QFileDialog, QMessageBox
        import pandas as pd
        import sqlite3

        # dossiers (adaptés à ton projet)
        img_folder = r"C:\Users\USER\Documents\Export"   # anciennes cartes (source)
        os.makedirs(PHOTOS_DIR, exist_ok=True)
        os.makedirs(CARDS_DIR, exist_ok=True)
        os.makedirs(PDFS_DIR, exist_ok=True)

        # sélection fichier Excel (tu voulais l'import manuel)
        excel_path, _ = QFileDialog.getOpenFileName(self, "Sélectionner le fichier Excel", "", "Excel Files (*.xlsx *.xls)")
        if not excel_path:
            return

        try:
            df = pd.read_excel(excel_path, dtype=str).fillna("")
        except Exception as e:
            QMessageBox.warning(self, "Erreur", f"Impossible de lire le fichier Excel : {e}")
            return

        conn = sqlite3.connect(STUDENTS_DB)
        c = conn.cursor()
        inserted = 0

        # conversion cm -> px (ajuste si besoin)
        px_per_cm = 37

        for i, row in df.iterrows():
            matricule = str(row.get("Matricule", "")).strip()
            if not matricule:
                continue

            # Construire rec tout de suite, photo vide par défaut
            rec = {
                "matricule": matricule,
                "nom": row.get("Nom", ""),
                "postnom": row.get("Post-Nom", ""),
                "prenom": row.get("Prénom", ""),
                "pere": row.get("Nom du Père", ""),
                "mere": row.get("Nom de la Mère", ""),
                "naissance": row.get("Naissance", ""),
                "sexe": row.get("Sexe", ""),
                "institution": row.get("Institution", ""),
                "classe": row.get("Classe", ""),
                "resultat": row.get("Resultat", ""),
                "mention": row.get("Mention", ""),
                "photo": "",  # on remplira avec le chemin complet si crop ok
                "owner": "admin",
                "date_enreg": datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
                "statut": "actif"
            }

            # --- Recherche de l'ancienne carte (ta méthode qui fonctionnait) ---
            old_card_path = ""
            for f in os.listdir(img_folder):
                fname, ext = os.path.splitext(f)
                if ext.lower() not in [".jpg", ".jpeg", ".png", ".bmp"]:
                    continue
                if matricule and fname.startswith(matricule):
                    old_card_path = os.path.join(img_folder, f)
                    break
                nom_complet = (str(row.get("Nom","")).strip() + str(row.get("Post-Nom","")).strip() + str(row.get("Prénom","")).strip()).lower()
                if nom_complet and nom_complet in fname.lower():
                    old_card_path = os.path.join(img_folder, f)
                    break

            photo_path = ""  # par défaut vide si pas trouvé / crop fail

            if old_card_path and os.path.exists(old_card_path):
                try:
                    img = cv2.imread(old_card_path)
                    if img is None:
                        raise ValueError("Image illisible")

                    h_img, w_img = img.shape[:2]

                    # --- Crop actuel (déjà présent dans le code) ---
                    X = max(w_img - 350, 0)
                    Y = max(50, 0)
                    W = 250
                    H = 350

                    # --- Ajustements supplémentaires relatifs au crop actuel ---
                    px_per_cm = 37  # conversion cm -> pixels
                    extra_left_cm = 1.2  # rogner à gauche
                    extra_bottom_cm = 0  # rogner en bas

                    extra_left_px = int(round(extra_left_cm * px_per_cm))
                    extra_bottom_px = int(round(extra_bottom_cm * px_per_cm))

                    # Calcul du nouveau crop basé sur l'ancien
                    X_new = X + extra_left_px                # décalage à droite pour rogner la gauche
                    Y_new = Y                                # pas de changement en haut
                    W_new = W - extra_left_px                # largeur réduite pour rogner gauche
                    H_new = H - extra_bottom_px              # hauteur réduite pour rogner en bas

                    # Clamp pour rester dans l'image
                    X_new = max(0, X_new)
                    Y_new = max(0, Y_new)
                    W_new = min(W_new, w_img - X_new)
                    H_new = min(H_new, h_img - Y_new)

                    # Crop final
                    crop = img[Y_new:Y_new+H_new, X_new:X_new+W_new]
                    # --- Nouveau crop sur le crop existant ---
                    extra_left_cm = 1.5    # rogner à gauche
                    extra_bottom_cm = 0  # rogner en bas

                    extra_left_px = int(round(extra_left_cm * px_per_cm))
                    extra_bottom_px = int(round(extra_bottom_cm * px_per_cm))

                    # Calcul des nouvelles dimensions basées sur le crop actuel
                    X_final = extra_left_px
                    Y_final = 0  # pas de changement en haut
                    W_final = crop.shape[1] - extra_left_px
                    H_final = crop.shape[0] - extra_bottom_px

                    # Clamp pour rester dans l'image
                    X_final = max(0, X_final)
                    Y_final = max(0, Y_final)
                    W_final = max(1, W_final)
                    H_final = max(1, H_final)

                    # Crop final
                    crop = crop[Y_final:Y_final+H_final, X_final:X_final+W_final]
                    # --- Troisième crop sur le crop existant ---
                    third_top_cm = 0.8      # rogner en haut
                    third_bottom_cm = 2.2   # rogner en bas
                    third_left_cm = 0    # pas de changement à gauche
                    third_right_cm = 0    # rogner à droite

                    third_top_px = int(round(third_top_cm * px_per_cm))
                    third_bottom_px = int(round(third_bottom_cm * px_per_cm))
                    third_left_px = int(round(third_left_cm * px_per_cm))
                    third_right_px = int(round(third_right_cm * px_per_cm))

                    X_third = third_left_px
                    Y_third = third_top_px
                    W_third = crop.shape[1] - (third_left_px + third_right_px)
                    H_third = crop.shape[0] - (third_top_px + third_bottom_px)

                    # Clamp pour rester dans l'image
                    X_third = max(0, X_third)
                    Y_third = max(0, Y_third)
                    W_third = max(1, W_third)
                    H_third = max(1, H_third)

                    # Crop final
                    crop = crop[Y_third:Y_third+H_third, X_third:X_third+W_third]

                    # Redimension final à 300x400 (portrait) — déformation possible pour remplir tout l'espace
                    crop_resized = cv2.resize(crop, (300, 400), interpolation=cv2.INTER_LANCZOS4)

                    # Sauvegarde de la photo finale
                    photo_fname = f"{matricule}.jpg"
                    photo_path = os.path.join(PHOTOS_DIR, photo_fname)
                    cv2.imwrite(photo_path, crop_resized)

                    # Mettre le chemin complet dans rec
                    rec["photo"] = photo_path

                except Exception as e:
                    print(f"Erreur crop photo {matricule}: {e}")
                    rec["photo"] = ""
            else:
                print(f"⚠️ Ancienne carte non trouvée pour {matricule} (checked folder: {img_folder})")


            # --- Insertion / update DB ---
            try:
                cols = ",".join(rec.keys())
                vals = ",".join(["?"] * len(rec))
                c.execute(f"INSERT OR REPLACE INTO students ({cols}) VALUES ({vals})", tuple(rec.values()))
                inserted += 1
                conn.commit()
            except Exception as e:
                print(f"Erreur insertion {matricule}: {e}")
                # on continue quand même

            # --- Génération QR + carte + PDF ---
            try:
                qr_data = f"{rec['matricule']}|{rec['nom']} {rec['postnom']} {rec['prenom']}"
                # utilise qrcode.make qui est simple et fiable
                qr_img = qrcode.make(qr_data)
                qr_path = os.path.join(CARDS_DIR, f"{matricule}_qr.png")
                qr_img.save(qr_path)
                rec["qr_path"] = qr_path

                # generate_id_card doit lire rec['photo'] (chemin complet) — vérifie qu'elle l'utilise
                generate_id_card(rec)
                generate_pdf(rec)

            except Exception as e:
                print(f"Erreur génération carte {matricule}: {e}")

        conn.close()
        QMessageBox.information(self, "Import terminé", f"{inserted} cartes traitées avec succès.\nPhotos recadrées dans: {PHOTOS_DIR}\nCartes: {CARDS_DIR}")
        # reload view
        self.load_table("students")


    def generate_id_card(rec):
        """
        Génère la carte d'identité moderne pour un étudiant.
        rec: dictionnaire contenant au moins
             'photo', 'qr_path', 'nom', 'postnom', 'prenom', 'matricule', 'classe'
        """
        try:
            # Dimensions
            CARD_W, CARD_H = 860, 540
            PHOTO_W, PHOTO_H = 400, 300
            BAND_H = 50  # hauteur bandes colorées

            # Créer carte blanche
            card = Image.new("RGB", (CARD_W, CARD_H), "white")
            draw = ImageDraw.Draw(card)

            # Bande verte en haut
            draw.rectangle([0, 0, CARD_W, BAND_H], fill=(0, 128, 0))

            # Bande rouge en bas
            draw.rectangle([0, CARD_H-BAND_H, CARD_W, CARD_H], fill=(200, 0, 0))

            # Ajouter texte étudiant
            font_path = "arial.ttf"  # Remplacer par chemin de police si besoin
            try:
                font_title = ImageFont.truetype(font_path, 36)
                font_sub = ImageFont.truetype(font_path, 28)
            except:
                font_title = ImageFont.load_default()
                font_sub = ImageFont.load_default()

            text_top = f"{rec['nom']} {rec['postnom']} {rec['prenom']}"
            draw.text((20, BAND_H + 10), text_top, fill="black", font=font_title)
            draw.text((20, BAND_H + 60), f"Matricule: {rec['matricule']}", fill="black", font=font_sub)
            draw.text((20, BAND_H + 100), f"Classe: {rec.get('classe','')}", fill="black", font=font_sub)

            # Charger photo
            photo_path = os.path.join(PHOTOS_DIR, rec['photo'])
            photo = Image.open(photo_path).convert("RGB")
            photo = photo.resize((PHOTO_W, PHOTO_H))
            photo_x = (CARD_W - PHOTO_W) // 2
            photo_y = CARD_H - BAND_H - PHOTO_H - 10  # 10px marge au dessus bande rouge
            card.paste(photo, (photo_x, photo_y))

            # Charger QR code
            qr = Image.open(rec['qr_path']).convert("RGB")
            qr_size = 120
            qr = qr.resize((qr_size, qr_size))
            qr_x = (CARD_W - qr_size) // 2
            qr_y = photo_y + PHOTO_H + 5  # 5px marge au dessus bande rouge
            card.paste(qr, (qr_x, qr_y))

            # Sauvegarder carte
            card_path = os.path.join(CARDS_DIR, f"{rec['matricule']}_card.jpg")
            card.save(card_path)

            # Ajouter le chemin dans rec pour référence
            rec['card_path'] = card_path

        except Exception as e:
            print(f"Erreur génération carte design pour {rec['matricule']}: {e}")
    from PIL import Image, ImageDraw, ImageFont

    def generate_id_card(rec):
        try:
            # Création de la carte vide
            card_width, card_height = 860, 540
            card = Image.new("RGB", (card_width, card_height), color=(255, 255, 255))
            draw = ImageDraw.Draw(card)

            # Couleurs bandes
            top_color = (0, 128, 0)     # vert
            bottom_color = (200, 0, 0)  # rouge
            top_height = 50
            bottom_height = 50

            # Bande verte en haut
            draw.rectangle([0, 0, card_width, top_height], fill=top_color)
            # Bande rouge en bas
            draw.rectangle([0, card_height - bottom_height, card_width, card_height], fill=bottom_color)

            # Charger photo
            photo_path = os.path.join(PHOTOS_DIR, rec["photo"])
            if os.path.exists(photo_path):
                photo = Image.open(photo_path)
                photo = photo.resize((300, 400))  # s'assure du bon format
                card.paste(photo, (50, top_height + 20))  # position photo

            # Charger QR
            qr_path = rec.get("qr_path", "")
            if qr_path and os.path.exists(qr_path):
                qr = Image.open(qr_path)
                qr = qr.resize((120, 120))
                card.paste(qr, (card_width - 50 - 120, card_height - bottom_height - 120))

            # Ajouter texte
            font_path = "arial.ttf"  # ou path vers une police dispo
            try:
                font_title = ImageFont.truetype(font_path, 30)
                font_text = ImageFont.truetype(font_path, 24)
            except:
                font_title = ImageFont.load_default()
                font_text = ImageFont.load_default()

            # Nom complet
            full_name = f"{rec['nom']} {rec['postnom']} {rec['prenom']}"
            draw.text((380, 100), full_name, font=font_title, fill=(0, 0, 0))

            # Institution + Classe
            draw.text((380, 150), f"Institution: {rec['institution']}", font=font_text, fill=(0, 0, 0))
            draw.text((380, 190), f"Classe: {rec['classe']}", font=font_text, fill=(0, 0, 0))
            draw.text((380, 230), f"Sexe: {rec['sexe']}", font=font_text, fill=(0, 0, 0))
            draw.text((380, 270), f"Naissance: {rec['naissance']}", font=font_text, fill=(0, 0, 0))
            draw.text((380, 310), f"Résultat: {rec['resultat']}", font=font_text, fill=(0, 0, 0))
            draw.text((380, 350), f"Mention: {rec['mention']}", font=font_text, fill=(0, 0, 0))

            # Sauvegarde carte
            card_filename = f"{rec['matricule']}_card.jpg"
            card_path = os.path.join(CARDS_DIR, card_filename)
            card.save(card_path)
            rec["card_path"] = card_path

        except Exception as e:
            print(f"Erreur création carte {rec['matricule']}: {e}")

    # ------------------- Settings -------------------
    def show_settings(self):
        self.current_table = "settings"
        self.set_active_menu(self.btn_settings)
        # cacher panels d'actions
        for b in self.status_buttons: b.setVisible(False)
        self.btn_delete_student.setVisible(False)
        for b in [self.btn_add_user, self.btn_edit_user, self.btn_delete_user, self.btn_suspend_user]:
            b.setVisible(False)

        self.table.clear()
        self.table.setRowCount(0)
        self.table.setColumnCount(0)
        QMessageBox.information(self, "Paramètres", "Gérer utilisateurs via l'onglet Users.")

# ------------------------------
# Run standalone
# ------------------------------
if __name__ == "__main__":
    app = QApplication(sys.argv)
    w = AdminPanel()
    w.show()
    sys.exit(app.exec())
