# gui/admin_page.py
import os
import sqlite3
from PyQt6.QtWidgets import (
    QWidget, QVBoxLayout, QHBoxLayout, QLabel, QPushButton,
    QLineEdit, QFormLayout, QListWidget, QListWidgetItem, QMessageBox, QInputDialog
)
from PyQt6.QtCore import Qt
from utils import hash_password, verify_password

DB_USERS = os.path.join(os.getcwd(), "database", "users.db")
os.makedirs(os.path.join(os.getcwd(), "database"), exist_ok=True)

# ---------------------------
# Initialisation DB utilisateurs
# ---------------------------
def init_users_db():
    conn = sqlite3.connect(DB_USERS)
    c = conn.cursor()
    c.execute("""
        CREATE TABLE IF NOT EXISTS users (
            username TEXT PRIMARY KEY,
            password TEXT,
            role TEXT
        )
    """)
    # Création d'un admin par défaut si inexistant
    c.execute("SELECT * FROM users WHERE username='admin'")
    if not c.fetchone():
        c.execute("INSERT INTO users (username, password, role) VALUES (?, ?, ?)",
                  ("admin", hash_password("1234"), "admin"))
    conn.commit()
    conn.close()

init_users_db()


class AdminPage(QWidget):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("Administration - DLOGI APP")
        self.setMinimumSize(800, 600)
        self.init_ui()
        self.load_users()

    def init_ui(self):
        main_layout = QVBoxLayout(self)

        title_label = QLabel("Administration - DLOGI APP")
        title_label.setAlignment(Qt.AlignmentFlag.AlignCenter)
        title_label.setStyleSheet("font-size: 20px; font-weight: bold;")
        main_layout.addWidget(title_label)

        # ---------------------------
        # Boutons de navigation admin
        # ---------------------------
        btn_layout = QHBoxLayout()
        self.btn_students = QPushButton("DB Étudiants")
        self.btn_families = QPushButton("DB Familles")
        self.btn_nutrition = QPushButton("DB Nutrition")
        self.btn_params = QPushButton("Paramètres / Config")
        btn_layout.addWidget(self.btn_students)
        btn_layout.addWidget(self.btn_families)
        btn_layout.addWidget(self.btn_nutrition)
        btn_layout.addWidget(self.btn_params)
        main_layout.addLayout(btn_layout)

        # ---------------------------
        # Gestion utilisateurs
        # ---------------------------
        user_layout = QVBoxLayout()
        user_label = QLabel("Utilisateurs :")
        self.user_list = QListWidget()
        self.btn_add_user = QPushButton("Ajouter utilisateur")
        self.btn_edit_user = QPushButton("Modifier mot de passe")
        self.btn_delete_user = QPushButton("Supprimer utilisateur")

        user_layout.addWidget(user_label)
        user_layout.addWidget(self.user_list)
        user_btn_layout = QHBoxLayout()
        user_btn_layout.addWidget(self.btn_add_user)
        user_btn_layout.addWidget(self.btn_edit_user)
        user_btn_layout.addWidget(self.btn_delete_user)
        user_layout.addLayout(user_btn_layout)

        main_layout.addLayout(user_layout)

        # ---------------------------
        # Connexions
        # ---------------------------
        self.btn_add_user.clicked.connect(self.add_user)
        self.btn_edit_user.clicked.connect(self.edit_user_password)
        self.btn_delete_user.clicked.connect(self.delete_user)
        self.btn_students.clicked.connect(lambda: self.access_db("students"))
        self.btn_families.clicked.connect(lambda: self.access_db("families"))
        self.btn_nutrition.clicked.connect(lambda: self.access_db("nutrition"))
        self.btn_params.clicked.connect(self.open_params)

        # Style sombre
        self.setStyleSheet("""
            QWidget { background-color: #2c3e50; color: white; font-family: Arial; }
            QPushButton { background-color: #1abc9c; border: none; padding: 6px; font-weight: bold; }
            QPushButton:hover { background-color: #16a085; }
            QListWidget { background-color: #34495e; border: 1px solid #1abc9c; }
        """)

    def load_users(self):
        self.user_list.clear()
        conn = sqlite3.connect(DB_USERS)
        c = conn.cursor()
        c.execute("SELECT username, role FROM users ORDER BY username")
        for row in c.fetchall():
            item = QListWidgetItem(f"{row[0]} ({row[1]})")
            self.user_list.addItem(item)
        conn.close()

    def add_user(self):
        username, ok = QInputDialog.getText(self, "Ajouter utilisateur", "Nom d'utilisateur:")
        if not ok or not username.strip():
            return
        password, ok2 = QInputDialog.getText(self, "Mot de passe", "Mot de passe:", QLineEdit.EchoMode.Password)
        if not ok2 or not password:
            return
        role, ok3 = QInputDialog.getItem(self, "Rôle", "Rôle de l'utilisateur:", ["admin", "user"], 1, False)
        if not ok3:
            return

        conn = sqlite3.connect(DB_USERS)
        c = conn.cursor()
        try:
            c.execute("INSERT INTO users (username, password, role) VALUES (?, ?, ?)",
                      (username, hash_password(password), role))
            conn.commit()
            QMessageBox.information(self, "Succès", f"Utilisateur {username} ajouté avec succès.")
            self.load_users()
        except sqlite3.IntegrityError:
            QMessageBox.warning(self, "Erreur", "Utilisateur déjà existant.")
        finally:
            conn.close()

    def edit_user_password(self):
        selected = self.user_list.currentItem()
        if not selected:
            QMessageBox.warning(self, "Erreur", "Sélectionnez un utilisateur.")
            return
        username = selected.text().split()[0]
        password, ok = QInputDialog.getText(self, "Modifier mot de passe",
                                            f"Nouveau mot de passe pour {username}:",
                                            QLineEdit.EchoMode.Password)
        if ok and password:
            conn = sqlite3.connect(DB_USERS)
            c = conn.cursor()
            c.execute("UPDATE users SET password=? WHERE username=?", (hash_password(password), username))
            conn.commit()
            conn.close()
            QMessageBox.information(self, "Succès", f"Mot de passe modifié pour {username}.")

    def delete_user(self):
        selected = self.user_list.currentItem()
        if not selected:
            QMessageBox.warning(self, "Erreur", "Sélectionnez un utilisateur.")
            return
        username = selected.text().split()[0]
        if username == "admin":
            QMessageBox.warning(self, "Erreur", "Impossible de supprimer l'administrateur principal.")
            return
        reply = QMessageBox.question(self, "Confirmation",
                                     f"Voulez-vous vraiment supprimer {username}?",
                                     QMessageBox.StandardButton.Yes | QMessageBox.StandardButton.No)
        if reply == QMessageBox.StandardButton.Yes:
            conn = sqlite3.connect(DB_USERS)
            c = conn.cursor()
            c.execute("DELETE FROM users WHERE username=?", (username,))
            conn.commit()
            conn.close()
            self.load_users()

    def access_db(self, db_name):
        password, ok = QInputDialog.getText(self, "Mot de passe admin",
                                            f"Entrez le mot de passe administrateur pour accéder à {db_name}:",
                                            QLineEdit.EchoMode.Password)
        if not ok:
            return
        if not verify_password(password, "1234"):  # vérifier contre admin par défaut
            QMessageBox.warning(self, "Erreur", "Mot de passe incorrect.")
            return
        QMessageBox.information(self, "Accès autorisé", f"Accès à la DB {db_name} autorisé.")

    def open_params(self):
        QMessageBox.information(self, "Paramètres", "Interface de configuration et paramètres.")
