# excel_import.py
import os
import shutil
import pandas as pd
from datetime import datetime
from PyQt6.QtWidgets import QFileDialog, QMessageBox

from db_manager import insert_student, insert_family, insert_nutrition
from id_card import generate_id_card
from pdf_manager import generate_pdf

DB_DIR = os.path.join(os.getcwd(), "database")
os.makedirs(DB_DIR, exist_ok=True)

def import_excel(parent_widget=None):
    """
    Ouvre un fichier Excel, lit les données, génère ID CARD et PDF,
    puis insère les données dans la base (students, families, nutrition)
    """
    # Choisir fichier Excel
    file_path, _ = QFileDialog.getOpenFileName(parent_widget, "Sélectionner le fichier Excel", "", "Excel Files (*.xlsx *.xls)")
    if not file_path:
        return

    try:
        df = pd.read_excel(file_path)
    except Exception as e:
        QMessageBox.warning(parent_widget, "Erreur", f"Impossible de lire le fichier Excel:\n{str(e)}")
        return

    for idx, row in df.iterrows():
        # --------------------------
        # Déterminer le type d'enregistrement
        # --------------------------
        record_type = None
        if "Classe" in row and "Institution" in row:
            record_type = "student"
        elif "Responsable" in row or "Nom Famille" in row:
            record_type = "family"
        elif "Dossier_no" in row or "Parent" in row:
            record_type = "nutrition"
        else:
            continue  # ligne ignorée si type inconnu

        try:
            if record_type == "student":
                matricule = str(row.get("Matricule", f"STU-{int(datetime.now().timestamp())}")).strip()
                data = {
                    "matricule": matricule,
                    "nom": str(row.get("Nom", "")).strip(),
                    "post_nom": str(row.get("Post-Nom", "")).strip(),
                    "prenom": str(row.get("Prénom", "")).strip(),
                    "pere": str(row.get("Nom du Père", "")).strip(),
                    "mere": str(row.get("Nom de la Mère", "")).strip(),
                    "naissance": str(row.get("Naissance", "")).strip(),
                    "sexe": str(row.get("Sexe", "")).strip(),
                    "ecole": str(row.get("Institution", "")).strip(),
                    "classe": str(row.get("Classe", "")).strip(),
                    "resultat": str(row.get("Resultat", "")).strip(),
                    "mention": str(row.get("Mention", "")).strip(),
                    "photo_path": None
                }
                photo_path = str(row.get("PHOTO", "")).strip()
                if photo_path and os.path.exists(photo_path):
                    dest = os.path.join(DB_DIR, f"{matricule}_photo.png")
                    try:
                        shutil.copy(photo_path, dest)
                        data["photo_path"] = dest
                    except:
                        data["photo_path"] = None

                # Enregistrer dans DB
                insert_student(data)

                # Générer ID CARD
                card_out = os.path.join(DB_DIR, f"{matricule}_IDCARD.png")
                generate_id_card(data, card_out)

                # Générer PDF fiche complète
                pdf_out = os.path.join(DB_DIR, f"{matricule}_FICHE.pdf")
                generate_pdf(data, pdf_out)

            elif record_type == "family":
                matricule = str(row.get("Matricule", f"FAM-{int(datetime.now().timestamp())}")).strip()
                data = {
                    "matricule": matricule,
                    "nom_famille": str(row.get("Nom Famille", "")).strip(),
                    "responsable": str(row.get("Responsable", "")).strip(),
                    "adresse": str(row.get("Adresse", "")).strip(),
                    "zone": str(row.get("Zone", "")).strip(),
                    "nb_enfant": int(row.get("Nb_Enfant", 0)),
                    "categorie": str(row.get("Categorie", "")).strip(),
                    "debut": str(row.get("Debut", "")).strip(),
                    "fin": str(row.get("Fin", "")).strip(),
                    "photo_path": None
                }
                photo_path = str(row.get("PHOTO", "")).strip()
                if photo_path and os.path.exists(photo_path):
                    dest = os.path.join(DB_DIR, f"{matricule}_photo.png")
                    try:
                        shutil.copy(photo_path, dest)
                        data["photo_path"] = dest
                    except:
                        data["photo_path"] = None

                insert_family(data)

                # Générer ID CARD
                card_out = os.path.join(DB_DIR, f"{matricule}_IDCARD.png")
                generate_id_card(data, card_out)

                # Générer PDF fiche complète
                pdf_out = os.path.join(DB_DIR, f"{matricule}_FICHE.pdf")
                generate_pdf(data, pdf_out)

            elif record_type == "nutrition":
                matricule = str(row.get("Matricule", f"NUT-{int(datetime.now().timestamp())}")).strip()
                data = {
                    "matricule": matricule,
                    "nom": str(row.get("Nom", "")).strip(),
                    "parent": str(row.get("Parent", "")).strip(),
                    "adresse": str(row.get("Adresse", "")).strip(),
                    "dossier_no": str(row.get("Dossier_no", "")).strip(),
                    "poids": float(row.get("Poids", 0.0)),
                    "taille": float(row.get("Taille", 0.0)),
                    "signes": str(row.get("Signes", "")).strip(),
                    "date_enregistrement": str(row.get("Date_enregistrement", "")).strip(),
                    "photo_path": None
                }
                photo_path = str(row.get("PHOTO", "")).strip()
                if photo_path and os.path.exists(photo_path):
                    dest = os.path.join(DB_DIR, f"{matricule}_photo.png")
                    try:
                        shutil.copy(photo_path, dest)
                        data["photo_path"] = dest
                    except:
                        data["photo_path"] = None

                insert_nutrition(data)

                # Générer ID CARD
                card_out = os.path.join(DB_DIR, f"{matricule}_IDCARD.png")
                generate_id_card(data, card_out)

                # Générer PDF fiche complète
                pdf_out = os.path.join(DB_DIR, f"{matricule}_FICHE.pdf")
                generate_pdf(data, pdf_out)

        except Exception as e:
            print(f"Erreur ligne {idx}: {e}")
            continue

    QMessageBox.information(parent_widget, "Import Excel", f"Importation terminée et cartes/PDF générés dans {DB_DIR}")
