# db_manager.py
import os
import sqlite3
from datetime import datetime

DB_DIR = os.path.join(os.getcwd(), "database")
os.makedirs(DB_DIR, exist_ok=True)
DB_PATH = os.path.join(DB_DIR, "dlogi.db")


def init_db():
    conn = sqlite3.connect(DB_PATH)
    cur = conn.cursor()
    # Students
    cur.execute("""
        CREATE TABLE IF NOT EXISTS students (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            matricule TEXT UNIQUE,
            nom TEXT,
            post_nom TEXT,
            prenom TEXT,
            pere TEXT,
            mere TEXT,
            ecole TEXT,
            classe TEXT,
            resultat TEXT,
            mention TEXT,
            naissance TEXT,
            sexe TEXT,
            photo_path TEXT,
            created_at TEXT
        )
    """)
    # Families
    cur.execute("""
        CREATE TABLE IF NOT EXISTS families (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            matricule TEXT UNIQUE,
            nom_famille TEXT,
            responsable TEXT,
            adresse TEXT,
            zone TEXT,
            nb_enfant INTEGER,
            categorie TEXT,
            debut TEXT,
            fin TEXT,
            photo_path TEXT,
            created_at TEXT
        )
    """)
    # Nutrition
    cur.execute("""
        CREATE TABLE IF NOT EXISTS nutrition (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            matricule TEXT UNIQUE,
            nom TEXT,
            parent TEXT,
            adresse TEXT,
            dossier_no TEXT,
            poids REAL,
            taille REAL,
            signes TEXT,
            date_enregistrement TEXT,
            photo_path TEXT,
            created_at TEXT
        )
    """)
    # Access logs (QR scan)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS access_logs (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            matricule TEXT,
            type TEXT,  -- student/family/nutrition
            source TEXT,  -- camera, POS, autre
            timestamp TEXT
        )
    """)
    conn.commit()
    conn.close()


# -----------------------
# CRUD Functions
# -----------------------
def insert_student(data: dict):
    conn = sqlite3.connect(DB_PATH)
    cur = conn.cursor()
    try:
        cur.execute("""
            INSERT INTO students (matricule, nom, post_nom, prenom, pere, mere, ecole, classe, resultat, mention, naissance, sexe, photo_path, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        """, (
            data.get("matricule"),
            data.get("nom"),
            data.get("post_nom"),
            data.get("prenom"),
            data.get("pere"),
            data.get("mere"),
            data.get("ecole"),
            data.get("classe"),
            data.get("resultat"),
            data.get("mention"),
            data.get("naissance"),
            data.get("sexe"),
            data.get("photo_path"),
            datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        ))
        conn.commit()
    except sqlite3.IntegrityError:
        raise ValueError("Matricule déjà existant")
    finally:
        conn.close()


def insert_family(data: dict):
    conn = sqlite3.connect(DB_PATH)
    cur = conn.cursor()
    try:
        cur.execute("""
            INSERT INTO families (matricule, nom_famille, responsable, adresse, zone, nb_enfant, categorie, debut, fin, photo_path, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        """, (
            data.get("matricule"),
            data.get("nom_famille"),
            data.get("responsable"),
            data.get("adresse"),
            data.get("zone"),
            data.get("nb_enfant", 0),
            data.get("categorie"),
            data.get("debut"),
            data.get("fin"),
            data.get("photo_path"),
            datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        ))
        conn.commit()
    except sqlite3.IntegrityError:
        raise ValueError("Matricule déjà existant")
    finally:
        conn.close()


def insert_nutrition(data: dict):
    conn = sqlite3.connect(DB_PATH)
    cur = conn.cursor()
    try:
        cur.execute("""
            INSERT INTO nutrition (matricule, nom, parent, adresse, dossier_no, poids, taille, signes, date_enregistrement, photo_path, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        """, (
            data.get("matricule"),
            data.get("nom"),
            data.get("parent"),
            data.get("adresse"),
            data.get("dossier_no"),
            data.get("poids", 0.0),
            data.get("taille", 0.0),
            data.get("signes"),
            data.get("date_enregistrement"),
            data.get("photo_path"),
            datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        ))
        conn.commit()
    except sqlite3.IntegrityError:
        raise ValueError("Matricule déjà existant")
    finally:
        conn.close()


# -----------------------
# Search Functions
# -----------------------
def search_students(query: str):
    conn = sqlite3.connect(DB_PATH)
    cur = conn.cursor()
    cur.execute("""
        SELECT matricule, nom, post_nom, prenom, ecole, classe
        FROM students
        WHERE matricule LIKE ? OR nom LIKE ? OR prenom LIKE ?
        LIMIT 50
    """, (f"%{query}%", f"%{query}%", f"%{query}%"))
    res = cur.fetchall()
    conn.close()
    return res


def search_families(query: str):
    conn = sqlite3.connect(DB_PATH)
    cur = conn.cursor()
    cur.execute("""
        SELECT matricule, nom_famille, responsable
        FROM families
        WHERE matricule LIKE ? OR nom_famille LIKE ?
        LIMIT 50
    """, (f"%{query}%", f"%{query}%"))
    res = cur.fetchall()
    conn.close()
    return res


def search_nutrition(query: str):
    conn = sqlite3.connect(DB_PATH)
    cur = conn.cursor()
    cur.execute("""
        SELECT matricule, nom, dossier_no
        FROM nutrition
        WHERE matricule LIKE ? OR nom LIKE ?
        LIMIT 50
    """, (f"%{query}%", f"%{query}%"))
    res = cur.fetchall()
    conn.close()
    return res


# -----------------------
# Access log
# -----------------------
def log_access(matricule: str, type_: str, source: str):
    conn = sqlite3.connect(DB_PATH)
    cur = conn.cursor()
    cur.execute("""
        INSERT INTO access_logs (matricule, type, source, timestamp)
        VALUES (?, ?, ?, ?)
    """, (matricule, type_, source, datetime.now().strftime("%Y-%m-%d %H:%M:%S")))
    conn.commit()
    conn.close()


# Initialize DB at import
init_db()
